"""
screen_start.py
opening game screen
"""
import random
import pygame

import gfx, snd, hud, screen_menu
from map import Map




class Screen:
    def __init__(self, next_screen, player, opponent, interval=800.0):
        self.player = player
        self.opponent = opponent
        self.map = Map()
        self.mapdirty = True
        
        self.origbackground = gfx.load('playscreen')
        self.background = self.origbackground.convert()
        
        self.abort = 0
        self.winner = 0
        self.hiwall = None
        
        self.time = 0.0
        self.growinterval = 80.0
        self.watchinterval = interval
        self.lastgrow = 1.0
        self.lastwatch = 1.0
        
        self.watch = hud.Watch((530, 380))
        self.fx = hud.EffectsList()
        
        texture = 'texture'+str(random.randint(1,8))
        self.blankimage = gfx.loadcolored(texture, (120,120,120))
        



    def event(self, event):
        if event.type == pygame.KEYDOWN:
            if event.key == pygame.K_ESCAPE:
                self.abort = 1
        elif event.type == pygame.MOUSEMOTION:
            if self.player.getinput:
                self.hiwall = self.map.wallatpos(event.pos)
                if not self.map.wallavailable(self.hiwall):
                    self.hiwall = None
        elif event.type == pygame.MOUSEBUTTONDOWN:
            if self.hiwall:
                snd.play('tap')
                self.player.chosenwall = self.hiwall

        
        
    def update(self, time):
        self.time += time
        
        if self.abort:
            return screen_menu.Screen(None)
        
        if self.winner:
            if not len(self.fx.effects):
                return screen_menu.Screen(None)
            gfx.clear(self.background)
            #self.drawscorebar()
            self.fx.update(time)
            return self
 
        

        self.player.update(self.map)
        self.opponent.update(self.map)

        didbigtick = 0
        if int(self.lastwatch/self.watchinterval) != int(self.time/self.watchinterval):
            didbigtick = 1
            self.lastwatch += self.watchinterval
            if (self.player.chosenwall and self.opponent.chosenwall and
                        self.player.chosenwall == self.opponent.chosenwall):
                #print "PICKED THE SAME WALL, DISASTER STRIKES"
                self.map.fillwall(self.player.chosenwall, -1)
                self.player.chosenwall = self.opponent.chosenwall = None
                self.mapdirty = True
            if self.player.chosenwall:
                added = self.map.fillwall(self.player.chosenwall, 1)
                for c in added:
                    self.fx.append(hud.Glow(self.player.glows, c.rect))
                self.player.score += len(added)
                self.player.chosenwall = None
                self.mapdirty = True
            else:
                pos = random.randint(100, 150), random.randint(80, 100)
                self.fx.append(hud.Drift(self.player.missimg, pos, 400.0))
                snd.play('untap')
            if self.opponent.chosenwall:
                added = self.map.fillwall(self.opponent.chosenwall, 2)
                for c in added:
                    self.fx.append(hud.Glow(self.opponent.glows, c.rect))
                self.opponent.score += len(added)
                self.opponent.chosenwall = None
                self.mapdirty = True
            else:
                if self.opponent.playing:
                    pos = random.randint(530, 580), random.randint(80, 100)
                    self.fx.append(hud.Drift(self.opponent.missimg, pos, 400.0))
            
            self.player.newturn()
            self.opponent.newturn()
            if self.player.getinput:
                snd.play('tink')

        while self.time - self.lastgrow > self.growinterval:
            added = self.map.grow()
            if not didbigtick and (added[2] or added[3]):
                s = random.choice(['clink1', 'clink2', 'clink3', 'clink4'])
                snd.play(s)
            for c in added[2]:
                self.fx.append(hud.Glow(self.player.glows, c.rect))
            for c in added[3]:
                self.fx.append(hud.Glow(self.opponent.glows, c.rect))
            self.player.score += len(added[2])
            self.opponent.score += len(added[3])
            
            self.mapdirty = added[0] or added[2] or added[3] or self.mapdirty
            self.lastgrow += self.growinterval
        
        
        if self.mapdirty:
            self.mapdirty = False
            self.drawmap()
        gfx.clear(self.background)


        if self.hiwall:
            self.drawwallline(self.hiwall, self.player.color, 3)
        if self.player.chosenwall:
            self.drawwallline(self.player.chosenwall, self.player.color, 5)
        if self.opponent.chosenwall:
            self.drawwallline(self.opponent.chosenwall, self.opponent.color, 5)
        
        
        self.watch.draw(int(self.time/self.watchinterval)-(self.time/self.watchinterval))

        #self.drawscorebar()

        self.fx.update(time)

        #check for winner
        if self.player.score > self.opponent.score:
            if self.player.score > self.opponent.score + self.map.remaining:
                snd.play('clap')
                self.winner = 1
                self.fx.append(hud.Drift(self.player.winimg, (150, 280), 3000.0))
                self.fx.append(hud.Drift(self.opponent.loseimg, (490, 280), 2000.0))
        else:
            if self.opponent.playing and self.opponent.score > self.player.score + self.map.remaining:
                snd.play('clap')
                self.winner = 1
                self.fx.append(hud.Drift(self.player.loseimg, (150, 280), 2000.0))
                self.fx.append(hud.Drift(self.opponent.winimg, (490, 280), 3000.0))
        if not self.map.remaining and self.player.score == self.opponent.score:
            snd.play('clap')
            self.fx.append(hud.Drift(self.player.winimg, (150, 280), 3000.0))
            self.fx.append(hud.Drift(self.opponent.winimg, (490, 280), 3000.0))
            self.winner = -1


        #finish
        #if not self.buttons:
        #    return self.next_screen
        return self


    def drawmap(self):
        
        map = self.map
        minipic = pygame.Surface((153, 99))
        fill = minipic.fill
        fill((255, 0, 255))
        black = minipic.map_rgb((0, 0, 0))

        textures = self.blankimage, None, self.player.image, self.opponent.image
        for c in map.cells.itervalues():
            if c.state:
                tex = textures[c.state+1]
                minipic.blit(tex, c.rectmini, c.rectmini)

        line = pygame.draw.line
        for x in range(map.cellsx+1):
            for y in range(map.cellsy+1):
                c = x,y,0
                if map.walls.get(c, 0) > 0:
                    start, end = map.wallcoordsmini(c)
                    line(minipic, 0, start, end, 1)
                c = x,y,1
                if map.walls.get(c, 0) > 0:
                    start, end = map.wallcoordsmini(c)
                    line(minipic, black, start, end, 1)

        bigpic = pygame.transform.scale2x(minipic)
        bigpic = pygame.transform.scale2x(bigpic)
        bigpic.set_colorkey((255, 0, 255))
        
        self.background.blit(self.origbackground, (0, 0))
        self.drawscorebar()
        self.background.blit(bigpic, (map.offx-5, map.offy-5))

        set = self.background.set_at
        for cell in map.cells.itervalues():
            wall = cell.coord[0], cell.coord[1], 0
            if not self.map.walls[wall]:
                set(cell.rect.topleft, black)


    def drawwallline(self, wall, color, thick):
            start, end = self.map.wallcoords(wall)
            if wall[2]:
                start = start[0], start[1]+(7-thick)
                end = end[0], end[1]-(7-thick)
            else:
                start = start[0]+(7-thick), start[1]
                end = end[0]-(7-thick), end[1]
            c1 = color
            c2 = [max(0, c-90) for c in c1]
            pygame.draw.line(gfx.Surf, c2, start, end, thick)
            pygame.draw.line(gfx.Surf, c1, start, end, thick/2)

    
    def drawscorebar(self):
        #clear out old
        #self.background.blit(self.origbackground, (0, 0), (0, 0, 640, 120))
        blit = self.background.blit
        
        wasted = (self.map.totalcells-self.map.remaining)-(self.player.score + self.opponent.score)
        
        pct = float(self.player.score) / (self.map.totalcells-wasted)
        r = pygame.Rect(0, 0, int(pct*640), 120)
        blit(self.player.scorebar, r, r)
        
        pct = float(self.opponent.score) / (self.map.totalcells-wasted)
        r = pygame.Rect(0, 0, int(pct*640), 120)
        r.right = 640
        blit(self.opponent.scorebar, r, r)
        
        img = gfx.texta(80, self.player.hicolor, str(self.player.score), topleft=(20,3))
        blit(*img)
        if self.opponent.playing:
            img = gfx.texta(80, self.opponent.hicolor, str(self.opponent.score), topright=(620,6))
            blit(*img)

