"""
map.py
manage the grid map class
"""

import random
import pygame
import gfx

#wall states: -1=blocked, 0=open, 1=filled
#cell states: -1=blocked, 0=open, 1=player, 2=opponent

class Cell(object):
    def __init__(self, coord, state):
        x, y = coord
        self.coord = coord
        self.state = state
        self.walls = (x,y,0), (x,y,1), (coord[0],coord[1]+1,0), (coord[0]+1,coord[1],1)
        self.rect = pygame.Rect(coord[0]*24+20, coord[1]*24+70, 24, 24)
        self.rectmini = pygame.Rect(coord[0]*6+1, coord[1]*6+1, 6, 6)
    
    def coordoffset(self, x, y):
        return self.coord[0]+x, self.coord[1]+y




class Map(object):
    def __init__(self):
        self.offx = 20
        self.offy = 70
        self.cellsx = 25
        self.cellsy = 16
        self.cellz = 24 
        self.cells = {}
        self.walls = {}
        self.hotwalls = []
        self.remaining = self.cellsx * self.cellsy
        self.initmap()
        

    def cellatpos(self, pos):
        x = (pos[0]-self.offx)/24
        y = (pos[1]-self.offy)/24
        return self.cells.get((x,y))


    def wallatpos(self, pos):
        px = pos[0]-self.offx
        py = pos[1]-self.offy
        w = abs(12-((px-12)%24))
        h = abs(12-((py-12)%24))
        if w<h:            
            x = (px+10)/24
            y = (py+0)/24
        else:
            x = (px+0)/24
            y = (py+10)/24

        if not 0<=x<self.cellsx or not 0<=y<self.cellsy:
            return None

        return x, y, int(w<h)


    def wallavailable(self, wall):
        return self.walls.get(wall, 1) == 0


    def wallcoords(self, wall):
        start = wall[0]*24+self.offx, wall[1]*24+self.offy
        if wall[2]: #vert
            end = wall[0]*24+self.offx, (wall[1]+1)*24+self.offy
        else: #horz
            end = (wall[0]+1)*24+self.offx, wall[1]*24+self.offy
        return start, end

    def wallcoordsmini(self, wall):
        start = wall[0]*6+1, wall[1]*6+1
        if wall[2]: #vert
            end = wall[0]*6+1, (wall[1]+1)*6+1
        else: #horz
            end = (wall[0]+1)*6+1, wall[1]*6+1
        return start, end


    def initmap(self):
        for x in range(self.cellsx):
            for y in range(self.cellsy):
                coord = x, y
                self.cells[coord] = Cell(coord, 0)
        
        for x in range(self.cellsx+1):
            for y in range(self.cellsy+1):
                coord = x, y
                self.walls[x,y,0] = 0
                self.walls[x,y,1] = 0
                
        #borders
        for x in range(self.cellsx):
            self.walls[x,0,0] = 1
            self.walls[x,self.cellsy,0] = 1
            self.walls[x,self.cellsy,1] = -1
            self.walls[x,self.cellsy+1,1] = -1
        for y in range(self.cellsy):
            self.walls[0,y,1] = 1
            self.walls[self.cellsx,y,1] = 1
            self.walls[self.cellsx,y,0] = -1
            self.walls[self.cellsx+1,y,0] = -1

        #eat corners
        tl = (0,0), (1,0), (2,0), (3,0), (4,0), (5,0), (0,1), (1,1), (2,1)
        for x,y in tl:
            self.fillwall((x,y+1,0), -1)
            self.fillwall((x+1,y,1), -1)
            self.cells[x,y].state = -1
            self.walls[x,y,0] = -1
            self.walls[x,y,1] = -1
        tr = (24,0), (23,0), (22,0), (21,0), (20,0), (19,0), (18,0), (17,0), (24,1), (23,1), (22,1), (21,1), (24,2), (23,2)
        for x,y in tr:
            self.fillwall((x,y+1,0), 0)
            self.fillwall((x,y,1), 0)
            self.cells[x,y].state = -1
            self.walls[x,y,0] = -1
            self.walls[x+1,y,1] = -1
        br = (24,15), (23, 15), (22, 15), (21,15), (20,15), (24,14), (23, 14), (22, 14), (21,14), (20,14), (24,13), (23,13), (23,13), (22,13), (21,13), (24,12), (23, 12), (22,12)
        for x,y in br:
            self.fillwall((x,y,0), 0)
            self.fillwall((x,y,1), 0)
            self.cells[x,y].state = -1
            self.walls[x,y+1,0] = -1
            self.walls[x+1,y,1] = -1
        bl = (0, 15), (1, 15), (2,15), (3,15), (4,15), (5,15), (0,14), (1,14), (2,14), (3,14), (4,14), 
        for x,y in bl:
            self.fillwall((x,y,0), 0)
            self.fillwall((x+1,y,1), 0)
            self.cells[x,y].state = -1
            self.walls[x,y+1,0] = -1
            self.walls[x,y,1] = -1

        #count remaining cells
        self.remaining = 0
        for c in self.cells.itervalues():
            if not c.state:
                self.remaining += 1
            else:
                c.state = 0
        self.totalcells = self.remaining

        #random fill
        for v in range(600):
            x = random.randint(0, self.cellsx-1)
            y = random.randint(0, self.cellsy-1)
            z = random.randint(0, 1)
            if not self.walls[x,y,z]:
                self.fillwall((x,y,z), -1)

        self.hotwalls = []



    def cellclosed(self, cell):
        for c in cell.walls:
            if not self.walls[c]:
                return False
        return True

    def cellclosedthree(self, cell):
        count = 0
        openwall = None
        for c in cell.walls:
            if self.walls[c]:
                count += 1
            else:
                openwall = c
        if count == 3:
            return openwall
        return None

    def cellclosedcount(self, cell):
        count = 0
        openwall = None
        for c in cell.walls:
            if self.walls[c]:
                count += 1
            else:
                openwall = c
        return count
   
    def wallcells(self, wall):
        if wall[2]: #vert
            a = self.cells.get((wall[0]-1,wall[1]))
            b = self.cells.get((wall[0], wall[1]))
        else:
            a = self.cells.get((wall[0],wall[1]-1))
            b = self.cells.get((wall[0], wall[1]))
        return a, b
    
    
    def fillwall(self, wall, player):
        added = []
        if not self.walls.get(wall, 1):
            self.walls[wall] = 1
            a, b = self.wallcells(wall)
            if a and self.cellclosed(a):
                added.append(a)
                a.state = player
                self.hotwalls.append(wall)
            if b and self.cellclosed(b):
                added.append(b)
                b.state = player
                self.hotwalls.append(wall)

            if a:
                for wall in a.walls:
                    c1, c2 = self.wallcells(wall)
                    if c1 and c2 and c1.state and c1.state == c2.state:
                        self.walls[wall] = -1
            if b:
                for wall in b.walls:
                    c1, c2 = self.wallcells(wall)
                    if c1 and c2 and c1.state and c1.state == c2.state:
                        self.walls[wall] = -1
        self.remaining -= len(added)
        return added

    def grow(self):
        added = [[], [], [], []]
        oldhot = self.hotwalls
        self.hotwalls = []
        for wall in oldhot:
            c1, c2 = self.wallcells(wall)
            if not c1 or not c2:
                continue
            openwall = self.cellclosedthree(c1)
            if openwall:
                added[c2.state+1] += self.fillwall(openwall, c2.state)
            openwall = self.cellclosedthree(c2)
            if openwall:
                added[c1.state+1] += self.fillwall(openwall, c1.state)
        return added
