import pygame
from pygame.locals import *
from OpenGL.GL import *
from OpenGL.GLU import *
from math import ceil, log

from config import config

__all__ = [
    "GUITexture",
    "load_tex",
    "load_gui_tex",
    "text_tex",
    "gui_tex",
    "surf2tex",    
    ]

class GUITexture(object):
    def __init__(self, texid, tx,ty, xs,ys):
        self.texid = texid
        self.xs = xs
        self.ys = ys
        self.tx = tx
        self.ty = ty

    def draw(self, x,y, width=None):
        sx,sy = config.resolution
        if width is None:
            width = self.xs / float(sx)
        xs = self.xs
        ys = self.ys
        tx = self.tx
        ty = self.ty
        aspect = sx/float(sy)
        height = (float(ys)/xs)*width*aspect
        xx = x + width
        yy = y - height
        glColor3f(1,1,1)
        glBindTexture(GL_TEXTURE_2D, self.texid)
        glEnable(GL_TEXTURE_2D)
        glBegin(GL_QUADS)
        glTexCoord2f(0,0);glVertex3f(x,y,0)
        glTexCoord2f(tx,0);glVertex3f(xx,y,0)
        glTexCoord2f(tx,ty);glVertex3f(xx,yy,0)
        glTexCoord2f(0,ty);glVertex3f(x,yy,0)
        glEnd()
        glDisable(GL_TEXTURE_2D)
        glBindTexture(GL_TEXTURE_2D,0)
        
def load_tex(filename, texid=None):
    s = pygame.image.load(filename)
    return surf2tex(s,texid)

def load_gui_tex(filename, texid=None):
    s = pygame.image.load(filename)
    return gui_tex(s,texid)        

def text_tex(text, fontsize, color, texid=None):
    f = pygame.font.Font(None, fontsize)
    s = f.render(text, 1, color, (0,0,0))
    return gui_tex(s,texid)

def gui_tex(s, texid=None):
    x,y = s.get_size()
    l2 = log(2)
    xl,yl = ceil(log(x)/l2),ceil(log(y)/l2)
    x2,y2 = 2**xl,2**yl
    s2 = pygame.Surface((x2,y2), HWSURFACE)
    s2.fill((0,0,0))
    s2.blit(s,(0,0))
    texid = surf2tex(s2)
    return GUITexture(texid, x/x2,y/y2, x,y)        

def surf2tex(surface, texid=None):
    ix,iy = surface.get_size()
    bytes = pygame.image.tostring(surface, "RGBX")
    if texid is None:
        texid = glGenTextures(1)
    glBindTexture(GL_TEXTURE_2D, texid)
    glTexImage2D(GL_TEXTURE_2D, 0, 3, ix, iy, 0, GL_RGBA, GL_UNSIGNED_BYTE, bytes)
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP)
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP)
    if config.texture_filtering:
        filter = GL_LINEAR
    else:
        filter = GL_NEAREST
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, filter)
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, filter)
    glBindTexture(GL_TEXTURE_2D, 0)
    return texid    
