from __future__ import generators

import pygame
from pygame.locals import *
from OpenGL.GL import *
from OpenGL.GLU import *

from random import random,randrange
from math import floor, sin, cos, pi

from renderer import renderer
from models import *
from entity import Entity
from gameobjects import *

from config import config

FPS = 40

## button locations hardcoded to save time
BUTTON_BUILD_CANNON = (0.02,0.22,0.10,0.13)
BUTTON_BUILD_WALL = (0.11,0.22,0.20,0.13)

## gui mode
MODE_WALK = 0
MODE_AIM = 1
MODE_POWER = 2
MODE_BUILD = 3
MODE_PLACEMENT = 4
MODE_GAMEOVER = 5

def debug_print(s):
    return # disable debug
    if __debug__: print s

def wall_gen(l):
    i = 0
    while 1:
        try:
            yield wall_xy(i,l)
        except IndexError:
            raise StopIteration
        i += 1

def wall_xy(i,l):
    if i < l:
        return i,0
    if i < l*2-1:
        return l-1,i-l+1
    if i < l*3-2:
        return l*3-3-i,l-1
    if i >= l*4-4:
        raise IndexError,i
    return 0,l*4-4-i
    
def pick_point(x,y):
    """pick a point from the ground plane"""
    sx,sy = config.resolution
    y = sy-y
    m1 = glGetDoublev(GL_MODELVIEW_MATRIX)
    m2 = glGetDoublev(GL_PROJECTION_MATRIX)
    v = glGetIntegerv(GL_VIEWPORT)
    wx,wy,wz = gluUnProject(x,y,0.5, m1,m2,v)
    wx2,wy2,wz2 = gluUnProject(x,y,0.6,m1,m2,v)
    d = wy/(wy-wy2)
    return wx + (wx2-wx)*d, wz + (wz2-wz)*d

def check_button(button, gx, gy):
    x,y,xx,yy = button
    if gx >= x and gy <= y and gx <= xx and gy >= yy:
        return 1
    return 0
    
class SheepGame(object):

    def init(self):
        renderer.init()
        self.sheep = {}
        self.players = []
        self.structures = {}
        self.projectiles = {}
        self.explosions = {}
        self.map = {}
        self.cannon = None
        self.old_mouse = 0,0
        self.tick = 0

        self.ai_cannons = []

    def start(self):
        ## startup screen
        ## startup menu
        ## -> gameloop
        self.new_game()
        self.mainloop()

    def new_game(self):
        self.init_players()
        self.init_map()
        self.init_sheep()
        self.mode = MODE_WALK
        self.rot = 0 # aiming
        self.power = 0
        self.skip_mouse = 0     

    def init_players(self):
        pl = self.players = [Player(), Player()]
        e = pl[0].entity = Entity(FarmerModel)
        e.active = 1
        e.scale = 0.35
        e.set_xyz(-13,0,0)
        pl[0].fence = Fence()
        pl[0].money = 500
        renderer.update_money(500)

        e = pl[1].entity = Entity(FarmerModel2)
        e.active = 0 ## invisible AI player
        e.scale = 0.35
        e.set_xyz(13,0,0)
        pl[1].fence = Fence()
        pl[1].money = 500

    def init_sheep(self):
        for dx in (-10,10):
            for n in range(15):
                while 1:
                    x = random()*10-5+dx
                    z = random()*10-5
                    if self.maptile(x,z) is None:
                        break
                self.add_sheep(x,z)

    def init_map(self):
        dx = -16
        for pl in self.players:
            f = pl.fence
            for x,y in wall_gen(l=14):
                wall = f.add_wall(x+dx+0.5,y-7.5)
                self.map[x+dx,y-8] = wall
            dx = 2

        self.add_structure(Cannon, -10,-1)
        #self.add_structure(Cannon, -10,-5)
        #self.add_structure(Cannon, -10,3)
        c1 = self.add_structure(Cannon, 9,-1)
        self.ai_cannons.append(c1)

    def event_handling(self,get_event=pygame.event.get):
        pl = self.players[0]
        
        if get_event(QUIT):
            self.quit = 1

        ev = get_event(KEYDOWN)
        for e in ev:
            if e.key == K_ESCAPE:
                self.quit = 1
            if e.key == K_p: ## TODO:pause
                pass

        if self.mode == MODE_GAMEOVER:
            return

        ev = get_event(MOUSEBUTTONDOWN)
        sx,sy = config.resolution
        for e in ev:
            if self.mode == MODE_WALK:
                x,y = e.pos
                if y/float(sy) >= 0.7:
                    ## gui event
                    gx,gy = x/float(sx),1-y/float(sy)
                    debug_print("gui: %.3f,%.3f" % (gx,gy))
                    self.handle_gui_event(gx,gy)
                else:
                    wx,wz = pick_point(x,y)
                    if e.button == 1: ## move
                        pl.moving = 1
                        pl.dest = (wx,wz)
                    elif e.button == 3: ## activate/repair
                        self.on_ground_click(wx,wz)
            elif self.mode == MODE_AIM:
                if e.button == 1: ## power mode
                    self.set_power_mode()
                elif e.button == 3: ## cancel
                    self.set_walk_mode()
                    self.power = 0
            elif self.mode == MODE_PLACEMENT:
                if e.button == 1: ## place building
                    self.place_building()
                elif e.button == 3: ## cancel
                    self.set_walk_mode()

        ev = get_event(MOUSEBUTTONUP)
        if self.mode == MODE_POWER:
            if ev:
                #self.power = 0
                self.set_walk_mode()
                ## fire cannon
                self.launch_projectile()
            
        ev = get_event(MOUSEMOTION)
        if self.mode == MODE_AIM:
            for e in ev:
                if self.skip_mouse:
                    self.skip_mouse = 0
                    continue
                x,y = e.rel
                self.rot -= x*0.3
                self.cannon.entity.rot = self.rot

    def set_aim_mode(self):
        self.old_mouse = pygame.mouse.get_pos()
        pygame.event.set_grab(1)
        pygame.mouse.set_visible(0)
        self.mode = MODE_AIM

    def set_power_mode(self):
        pygame.event.set_grab(1)
        pygame.mouse.set_visible(0)
        self.mode = MODE_POWER

    def set_walk_mode(self):
        pygame.event.set_grab(0)
        pygame.mouse.set_visible(1)
        self.mode = MODE_WALK
        if self.old_mouse is not None:
            pygame.mouse.set_pos(self.old_mouse)

    def set_placement_mode(self, structure_cls):
        self.old_mouse = None
        self.mode = MODE_PLACEMENT
        self.structure_cls = structure_cls

    def handle_gui_event(self,gx,gy):        
        if check_button(BUTTON_BUILD_CANNON,gx,gy):
            self.on_cannon_clicked()
        if check_button(BUTTON_BUILD_WALL,gx,gy):
            self.on_wall_clicked()

    def on_cannon_clicked(self):
        debug_print("build cannon")
        self.set_placement_mode(Cannon)
    
    def on_wall_clicked(self):
        debug_print("build wall")
        ## TODO: metal walls
        self.set_placement_mode(MetalWall)

    def on_ground_click(self, wx,wz):
        wx,wz = floor(wx),floor(wz)
        debug_print("ground click %i,%i" % (wx,wz))
        tile = self.maptile(wx,wz)
        if tile is None:
            return
        tx,tz = floor(tile.entity.x)+0.5,floor(tile.entity.z)+0.5
        e = self.players[0].entity
        dist = abs(e.x-tx)+abs(e.z-tz)
        if dist > 2.0:
            return
        self.players[0].moving = 0
        if type(tile) == Cannon:
            self.on_cannon_activated(tile)
        elif type(tile) == WallPiece:
            self.on_wall_repair(tile)

    def on_cannon_activated(self, cannon):
        if cannon.cooldown > 0:
            return
        debug_print("cannon activated!")
        self.rot = cannon.entity.rot
        print self.rot
        self.power = 0
        self.set_aim_mode()
        self.skip_mouse = 1
        self.cannon = cannon

    def on_wall_repair(self, wall):
        debug_print("repair wall!")
        if wall.hp >= 1.0: return
        pl = self.players[0]
        cost = int((1.0-wall.hp) * 50)
        if pl.money < cost:
            repair = pl.money / 50.0
            pl.money = 0
            wall.hp += repair
        else:
            pl.money -= cost
            wall.hp = 1.0
        pl.fence.update_walls()
        renderer.update_money(pl.money)

    def give_income(self):
        pls = self.players
        lost0 = 1
        lost1 = 1
        for s in self.sheep.keys():
            e = s.entity
            if e.x < 0:
                pls[0].money += 7
                lost0 = 0
            else:
                pls[1].money += 7
                lost1 = 0
        renderer.update_money(pls[0].money)
        if lost0:
            self.you_lose()
        if lost1:
            self.you_win()

    def you_lose(self):
        #sys.exit(0) ## TODO!!
        self.mode = MODE_GAMEOVER
        renderer.set_msg("YOU LOSE!")

    def you_win(self):
        #sys.exit(0) ## TODO!!
        self.mode = MODE_GAMEOVER
        renderer.set_msg("YOU WIN!")

    def tick_ai(self):
        pl = self.players[1]
        for c in self.ai_cannons:
            if c.cooldown == 0 and random() < 0.02:
                self.ai_launch_projectile(c)
        if pl.money >= Cannon.price and random() < 0.01:
            self.ai_place_cannon()

        for w in pl.fence.walls:
            if w.hp <= 0.0 and random() < 0.001:
                self.ai_repair_wall(w)
            elif w.hp < 1.0 and random() < 0.0002:
                self.ai_repair_wall(w)

    def ai_repair_wall(self, wall):
        pl = self.players[1]
        cost = int((1.0-wall.hp) * 50)
        if pl.money < cost:
            repair = pl.money / 50.0
            pl.money = 0
            wall.hp += repair
        else:
            pl.money -= cost
            wall.hp = 1.0
        pl.fence.update_walls()

    def ai_place_cannon(self):
        while 1:
            x = randrange(5,10)
            z = randrange(-5,5)
            if self.maptile(x,z) is None:
                break
        s = self.add_structure(Cannon, x,z)
        self.ai_cannons.append(s)
        self.players[1].money -= Cannon.price


    def ai_launch_projectile(self, cannon):
        e = cannon.entity
        e.rot = random()*60-90-30
        cannon.cooldown = 400
        real_pow = random()*0.6+0.3
        r = pi * (e.rot+180) / 180.0
        xx = -sin(r) * real_pow * 0.3
        zz = -cos(r) * real_pow * 0.3
        p = Projectile(e.x,e.z,xx,zz)
        self.projectiles[p] = None

    
    def action_handling(self):
        """move stuff around etc."""
        if (self.tick % 200 == 0):
            self.give_income()
        if self.mode == MODE_POWER:
            self.power += 0.05
        for pl in self.players:
            if pl.moving:
                dx,dz = pl.dest
                e = pl.entity
                e.face(dx,dz)
                e.forward(0.1)
                if self.maptile(e.x,e.z) is not None:
                    e.forward(-0.1)
                    pl.moving = 0
                    continue
                d = abs(e.x-dx)+abs(e.z-dz)
                if d < 0.3:
                    pl.moving = 0

    def sheep_handling(self):
        pls = self.players
        for s in self.sheep.keys():
            e = s.entity
            if not (pls[0].fence.inside(e.x,e.z) or pls[1].fence.inside(e.x,e.z)):
                if e.x < 0: # bonus for sheep escape
                    pls[1].money += 200
                else:
                    pls[0].money += 200
                    renderer.update_money(pls[0].money)
                del self.sheep[s]
                continue
            if s.escaping:
                fx,fz = e.forward_xz(0.2)
                t = self.maptile(fx,fz)
                if isinstance(t,WallPiece) and t.hp > 0.0:
                    s.moving = 0
                    s.escaping = 0
                    continue
                e.forward(0.2)
                continue
            if s.moving:
                dx,dz = s.dest
                e.face(dx,dz)
                fx,fz = e.forward_xz(0.5)
                t = self.maptile(fx,fz)
                if t is not None and (not isinstance(t,WallPiece) or t.hp > 0.0):
                    s.moving = 0
                    continue
                e.forward(0.1)
                d = abs(e.x-dx)+abs(e.z-dz)
                if d < 0.5:
                    s.moving = 0
            if random() < 0.01:
                s.dest = (e.x+random()*8-4,e.z+random()*8-4)
                s.moving = 1
            if random() < 0.001:
                if e.x < 0:
                    walls = pls[0].fence.walls
                else:
                    walls = pls[1].fence.walls
                for w in walls:
                    if w.hp <= 0:
                        s.escaping = 1
                        e.face(w.entity.x,w.entity.z)
                        break

    def maptile(self, x, z):
        return self.map.get((floor(x),floor(z)),None)

    def add_sheep(self, x, z):
        self.sheep[Sheep(x,z)] = None

    def add_structure(self, cls, x, z):
        s = cls(x+.5,z+.5)
        self.structures[s] = None
        self.map[x,z] = s
        return s

    def launch_projectile(self):
        e = self.cannon.entity
        self.cannon.cooldown = 400
        real_pow = self.power % 2.0
        if real_pow > 1.0:
            real_pow = 2.0 - real_pow
        r = pi * (e.rot+180) / 180.0
        xx = -sin(r) * real_pow * 0.3
        zz = -cos(r) * real_pow * 0.3
        p = Projectile(e.x,e.z,xx,zz)
        self.projectiles[p] = None

    def projectile_handling(self):
        for p in self.projectiles.keys():
            p.tick()
            e = p.entity
            if e.y < 0:
                ## TODO: boom!
                del self.projectiles[p]
                e.y = 0.1
                self.damage(e.x,e.z)
                e = Explosion(e.x,e.y,e.z)
                self.explosions[e] = None

        for e in self.explosions.keys():
            e.tick()
            if e.entity.model.life > 100:
                del self.explosions[e]

    def structure_handling(self):
        for s in self.structures.keys():
            s.tick()

    def damage(self, ex,ey):
        tx = floor(ex)
        ty = floor(ey)
        wall0 = 0
        wall1 = 0
        for x in range(tx-2,tx+2):
            for y in range(ty-2,ty+2):
                t = self.maptile(x,y)
                if t is not None:
                    rng = (ex-x)**2 + (ey-y)**2
                    dam = 1.0/(1+rng)
                    if type(t) == WallPiece:
                        t.hp -= dam
                        debug_print("%.2f damage to %i,%i (%.2f left)" % (dam,x,y,t.hp))
                        if x < 0:
                            wall0 = 1
                        else:
                            wall1 = 1
                    elif type(t) == Cannon:
                        t.cooldown += int(dam*200)
        if wall0:
            self.players[0].fence.update_walls()
        if wall1:
            self.players[1].fence.update_walls()

    def mainloop(self):
        pygame.event.set_allowed(None)
        pygame.event.set_allowed([QUIT, KEYDOWN, MOUSEBUTTONDOWN, MOUSEBUTTONUP, MOUSEMOTION])

        clock = pygame.time.Clock()
        frames = 0
        ticks = 0
        self.quit = 0

        while not self.quit:
            t = pygame.time.get_ticks()
            renderer.init_frame()
            glTranslate(0,3,-25)
            glRotate(75,1,0,0)
            self.event_handling()
            if self.mode != MODE_GAMEOVER:
                self.tick += 1
                self.action_handling()
                self.tick_ai()
                self.sheep_handling()
                self.projectile_handling()
                self.structure_handling()
            self.draw_frame()
            t = pygame.time.get_ticks() - t
            ticks += t
            frames += 1
            clock.tick(FPS)

        print "avg. %.2fms per frame" % (ticks/float(frames))

    def placement_xz(self):
        pl = self.players[0]
        x,y = pygame.mouse.get_pos()
        wx,wz = pick_point(x,y)
        ox,oz = floor(pl.entity.x),floor(pl.entity.z)
        if wx < ox-1: wx = ox-1
        if wx > ox+1: wx = ox+1
        if wz < oz-1: wz = oz-1
        if wz > oz+1: wz = oz+1
        return floor(wx),floor(wz)

    def place_building(self):
        pl = self.players[0]
        e = pl.entity
        x,z = self.placement_xz()
        if pl.money < self.structure_cls.price:
            return
        if self.structure_cls == MetalWall:
            if type(self.maptile(x,z)) != WallPiece:
                return
        else:
            if self.maptile(x,z) is not None:
                return
            if floor(e.x) == x and floor(e.z) == z:
                return
        pl.money -= self.structure_cls.price
        renderer.update_money(pl.money)
        if self.structure_cls == MetalWall:
            t = self.maptile(x,z)
            pl.fence.walls.remove(t)
            w = pl.fence.add_wall(x+0.5,z+0.5,MetalWall)
            self.map[x,z] = w
            pl.fence.update_walls()
        else:
            self.add_structure(self.structure_cls, x, z)
        self.set_walk_mode()

    def draw_frame(self):

        renderer.draw_model(GroundModel(), (0,0,0))

        for s in self.sheep.keys():
            s.draw()

        for p in self.players:
            p.draw()
            p.fence.draw()

        for s in self.structures.keys():
            s.draw()

        for p in self.projectiles.keys():
            p.draw()

        if self.mode == MODE_PLACEMENT:
            x,z = self.placement_xz()
            e = self.structure_cls(x+0.5,z+0.5)
            e.draw()

        for e in self.explosions.keys():
            e.draw()

        real_pow = self.power % 2.0
        if real_pow > 1.0:
            real_pow = 2.0 - real_pow
        renderer.hud.power = real_pow
        renderer.hud.rot = self.rot
        renderer.draw_ui()
        
        glFlush()
        renderer.flip()
        

    def modelview(self, model):
        e = Entity(model)
        e.active = 1        
        pygame.event.set_allowed(None)
        pygame.event.set_allowed([QUIT])
        glClearColor(0,0,0,0)
        while not pygame.event.get():
            renderer.init_frame()
            glTranslate(0,0,-15)
            glRotate(40,1,0,0)
            glRotate(-40,0,1,0)
            glScale(2,2,2)
            e.draw()
            renderer.flip()
                                    
    def test(self):
        sheep = []
        s = Entity(FarmerModel)
        s.set_xyz(random()*10-15,1,random()*10-6)
        s.active = 1
        s.scale = 0.25
        sheep.append(s)
        s = Entity(FarmerModel2)
        s.set_xyz(random()*10+3,1,random()*10-6)
        s.active = 1
        s.scale = 0.25
        sheep.append(s)
        
        for n in range(5):
            s = Entity(SheepModel)
            s.set_xyz(random()*10-15,1,random()*10-6)
            s.active = 1
            s.scale = 0.15
            sheep.append(s)
        for n in range(5):
            s = Entity(SheepModel)
            s.set_xyz(random()*10+3,1,random()*10-6)
            s.active = 1
            s.scale = 0.15
            sheep.append(s)
        pygame.event.set_allowed(None)
        pygame.event.set_allowed([QUIT])
        clock = pygame.time.Clock()

        walls = []
        wallid = glGenLists(1)
        glNewList(wallid, GL_COMPILE)
        for x,y in wall_gen(l=14):
            wall = Entity(WallModel)
            wall.set_xyz(x-15.5,0,y-7)
            wall.active = 1
            wall.scale = 0.45
            walls.append(wall)
            wall.draw(cached=0)

        for x,y in wall_gen(l=14):
            wall = Entity(WallModel)
            wall.set_xyz(x+2.5,0,y-7)
            wall.active = 1
            wall.scale = 0.45
            walls.append(wall)
            wall.draw(cached=0)
        glEndList()

        frames = 0
        ticks = 0
        
        while not pygame.event.get():
            t = pygame.time.get_ticks()
            renderer.init_frame()
            glLoadIdentity()
            glTranslate(0,3,-25)
            glRotate(75,1,0,0)
            #glLightfv(GL_LIGHT1, GL_POSITION, (-10,10,-10,1))
            renderer.draw_model(GroundModel(), (0,0,0))
            for s in sheep:
                s.draw()
##            for w in walls:
##                w.draw()
            glCallList(wallid)
            renderer.draw_ui()
            glFlush()
            renderer.flip()
            for s in sheep:
                s.rotate(random()*10)
                s.forward(.1)
            t = pygame.time.get_ticks() - t
            ticks += t
            frames += 1
            clock.tick(FPS)

        print "avg. %.2fms per frame" % (ticks/float(frames))

def main():
    game = SheepGame()
    game.init()
    #game.modelview(MetalWallModel)
    game.start()
    #game.test()

if __name__ == '__main__':
    main()
