
import pygame
from pygame.locals import *
from OpenGL.GL import *
from OpenGL.GLU import *
from math import ceil, log, pi, sin, cos
from texutil import *

from config import config


class HUD(object):
    def __init__(self):
        self.power = 0
        self.rot = 0

class Renderer(object):
    def init(self):
        self.hud = HUD()
        flags = HWSURFACE|OPENGL|DOUBLEBUF
        if config.fullscreen:
            flags |= FULLSCREEN
        pygame.display.init()
        pygame.font.init()
        self.screen = pygame.display.set_mode(config.resolution, flags)
        pygame.display.set_caption("The Sheep Wars")
        self.init_gl()
        self.msg = None

    def init_gl(self):
        glClearColor(0.1,0.2,0.3,0)
        self.clear_matrices()
        self.init_projection()
        glShadeModel(GL_SMOOTH)
        glMatrixMode(GL_MODELVIEW)
        self.init_lights()
        self.init_textures()
        #glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)        

    def clear_matrices(self):
        old_mode = glGetIntegerv(GL_MATRIX_MODE)
        glMatrixMode(GL_PROJECTION)
        glLoadIdentity()
        glMatrixMode(GL_MODELVIEW)
        glLoadIdentity()
        glMatrixMode(old_mode)

    def init_projection(self):
        old_mode = glGetIntegerv(GL_MATRIX_MODE)
        glMatrixMode(GL_PROJECTION)
        x,y = config.resolution
        gluPerspective(config.fov, x/float(y), 0.01, config.view_distance)
        glMatrixMode(GL_MODELVIEW)
        glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST)
        glClearDepth(1)
        glDepthFunc(GL_LEQUAL)
        glEnable(GL_DEPTH_TEST)
        glMatrixMode(old_mode)

    def init_lights(self):
        glLightfv(GL_LIGHT1, GL_QUADRATIC_ATTENUATION, .005)
        glLightfv(GL_LIGHT1, GL_DIFFUSE, (1,1,1,1))
        glLightfv(GL_LIGHT1, GL_AMBIENT, (1,1,1,1))
        glLightfv(GL_LIGHT1, GL_POSITION, (0,4,0,1))
        glEnable(GL_LIGHT1)
        glEnable(GL_LIGHTING)

    def init_textures(self):

        self.ground_tex = load_tex("grass_tex.bmp")
        self.wood_tex = load_tex("wood_tex_low.bmp")
        self.wood_tex2 = load_tex("wood_tex2_low.bmp")
        self.wool_tex = load_tex("wool_tex.bmp")
        self.metal_tex = load_tex("metalplate_tex.bmp")

        self.gui_tex1 = load_gui_tex("gui1.bmp")
        self.gui_money = text_tex("$400", 30, (0,255,0))
        self.gui_title = load_gui_tex("title_tex.bmp")
        self.gui_hud = load_gui_tex("cannon_hud.bmp")

        self.gui_cannon_price = text_tex("$500", 16, (255,0,0))
        self.gui_wall_price = text_tex("$200", 16,(255,0,0))

    def init_frame(self):
        glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT)
        glLoadIdentity()

    def set_msg(self, msg):
        self.msg = text_tex(msg, 80, (255,0,0))

    def draw_model(self, model, (x,y,z), rot=0, scale=1.0, cached=1):
        glPushMatrix()
        glTranslatef(x,y,z)
        glRotatef(rot, 0,1,0)
        glScalef(scale,scale,scale)
        if cached:
            model.draw_cached()
        else:
            model.draw()
        glPopMatrix()

    def update_money(self, i):
        self.gui_money = text_tex("$%i"%i, 30, (0,255,0))

    def draw_ui(self):
        glPushAttrib(GL_ENABLE_BIT)
        glDisable(GL_LIGHTING)
        glDisable(GL_DEPTH_TEST)
        old_mode = glGetIntegerv(GL_MATRIX_MODE)
        glMatrixMode(GL_MODELVIEW)
        glPushMatrix()
        glLoadIdentity()
        glMatrixMode(GL_PROJECTION)
        glPushMatrix()
        glLoadIdentity()
        ## TODO: gui texture
        glOrtho(0,1,0,1,-1,1)        

        glBegin(GL_QUADS)
        glColor3f(0,0,0)
        glVertex3f(0,0.3,0)
        glVertex3f(1,0.3,0)
        glVertex3f(1,0,0)
        glVertex3f(0,0,0)
        glEnd()
        
        self.gui_title.draw(0.3,0.3,0.4)
        self.gui_tex1.draw(0.01,0.3,0.3)
        self.gui_money.draw(0.85,0.3)
        self.gui_hud.draw(0.3,0.22,0.2)

        if self.msg is not None:
            self.msg.draw(0.3,0.6)

        self.gui_cannon_price.draw(0.03,0.12,0.05)
        self.gui_wall_price.draw(0.13,0.12,0.05)

        glColor3f(255,0,0)

        x,y = 0.366,0.129
        rot = self.hud.rot * pi / 180.0
        xx = sin(rot)*0.05
        yy = cos(rot)*0.05

        glBegin(GL_LINES)
        glVertex2f(x,y)
        glVertex2f(x+xx,y-yy)
        glEnd()

        pow = self.hud.power
        x,y = 0.450,0.067
        xx,yy = 0.474,y+0.126*pow
        glBegin(GL_QUADS)
        glVertex2f(x,y)
        glVertex2f(xx,y)
        glVertex2f(xx,yy)
        glVertex2f(x,yy)
        glEnd()
        
        glPopMatrix()
        glMatrixMode(GL_MODELVIEW)
        glPopMatrix()
        glMatrixMode(old_mode)
        glPopAttrib()

    def flip(self):
        pygame.display.flip()

renderer = Renderer()
