
from entity import Entity
from models import *
from OpenGL.GL import *
from random import random, randrange

__all__ = [
    "Sheep",
    "WallPiece",
    "MetalWall",
    "Fence",
    "Player",
    "Cannon",
    "Projectile",
    "Explosion",
    ]

class GameObject(object):

    def draw(self, cached=0):
        self.entity.draw(cached)

    def tick(self): pass

class Sheep(GameObject):
    """wandering fluffy thing"""
    def __init__(self, x,z):
        e = self.entity = Entity(SheepModel)
        e.set_xyz(x,0,z)
        e.active = 1
        e.scale = 0.3
        e.rot = randrange(360)
        self.dest = (0,0)
        self.moving = 0
        self.escaping = 0

class WallPiece(GameObject):
    """one piece of a fence"""
    _model = WoodWallModel
    _scale = 0.4
    
    def __init__(self,x,z):
        self.hp = 1.0
        e = self.entity = Entity(self._model)
        e.set_xyz(x,0,z)
        e.active = 1
        e.scale = self._scale

class MetalWall(WallPiece):
    _model = MetalWallModel
    _scale = 0.45
    price = 200
    

class Fence(GameObject):
    """takes care of the display list for the whole fence object"""
    def __init__(self):
        self.walls = []
        self.listid = None
        self.minx = None
        self.miny = None
        self.maxx = None
        self.maxy = None

    def add_wall(self, x, y, cls=WallPiece):
        if self.minx is None: self.minx = x
        if self.miny is None: self.miny = y
        if self.maxx is None: self.maxx = x
        if self.maxy is None: self.maxy = y
        if x < self.minx: self.minx = x
        if x > self.maxx: self.maxx = x
        if y < self.miny: self.miny = y
        if y > self.maxy: self.maxy = y
        w = cls(x,y)
        self.walls.append(w)
        return w

    def inside(self, x, y):        
        return x > self.minx and x < self.maxx and y > self.miny and y < self.maxy

    def draw(self):
        if self.listid is None:
            id = self.listid = glGenLists(1)
            glNewList(id,GL_COMPILE_AND_EXECUTE)
            self.draw_walls()
            glEndList()
        else:
            glCallList(self.listid)

    def update_walls(self):
        glNewList(self.listid, GL_COMPILE)
        self.draw_walls()
        glEndList()

    def draw_walls(self):
        for w in self.walls:
            if type(w) == MetalWall:
                m_cls = MetalWallModel
                w.hp = 1.0
            else:
                m_cls = WoodWallModel
                if w.hp <= 0.0:
                    m_cls = BrokenWallModel
                    w.hp = 0.0
                elif w.hp < 1.0:
                    r = w.hp*0.8+0.2
                    #glMaterial(GL_FRONT_AND_BACK, GL_EMISSION, (r,0,0))
                    glDisable(GL_LIGHTING)
                    glColor3f(r,r,r)
            w.entity.model = m_cls()
            w.draw(cached=0)
            glEnable(GL_LIGHTING)
            #glMaterial(GL_FRONT_AND_BACK, GL_EMISSION, (0,0,0))

class Cannon(GameObject):
    """a turret used for destroying opponent's fence"""

    price = 500
    
    def __init__(self,x,z):
        self.hp = 1.0
        e = self.entity = Entity(CannonModel)
        e.set_xyz(x,0,z)
        e.active = 1
        e.scale = 0.4
        self.cooldown = 0

    def tick(self):
        if self.cooldown > 0:
            self.cooldown -= 1

    def draw(self,cached=0):
        if self.cooldown > 0:
            r = self.cooldown * 0.01
            glMaterial(GL_FRONT_AND_BACK, GL_EMISSION, (r,0,0))
        self.entity.draw(cached)
        glMaterial(GL_FRONT_AND_BACK, GL_EMISSION, (0,0,0))

class Player(GameObject):
    def __init__(self):
        self.money = 0  # used to build resources
        self.fence = None # status of walls
        self.dest = (0,0)
        self.moving = 0

class Projectile(GameObject):
    def __init__(self, x, z, xx, zz):
        e = self.entity = Entity(CannonBallModel)
        e.set_xyz(x+xx,0.5,z+zz)
        e.active = 1
        e.scale = 0.4
        self.xx = xx
        self.zz = zz
        self.yy = 0.2

    def tick(self):
        e = self.entity
        self.yy -= 0.004
        e.x += self.xx
        e.y += self.yy
        e.z += self.zz
        
class Explosion(GameObject):
    def __init__(self, x, y, z):
        e = self.entity = Entity(ExplosionModel)
        e.set_xyz(x,y,z)
        e.active = 1

    def tick(self):
        e = self.entity
        e.model.life += 4
