#! /usr/bin/env python
# $Id: sheep.py,v 1.12 2002/11/11 00:42:01 richard Exp richard $

# TODO later
# - sound [no recording equipment :(]
# - show sheep damaging walls
# - show player fixing walls
# - proper path-finding for sheep

import sys, time, random, math, os
import weakref

import pygame, pygame.font
from pygame.locals import *
pygame.init()

# wah, can't use the cool modenine font :(
font = pygame.font.Font(None, 30)

import maps

def magnitude(x, y):
    return math.sqrt(x*x + y*y)

def normalise(x, y):
    d = math.sqrt(x*x + y*y)
    return x/d, y/d

def renderText(surf, x, y, text):
    sx,sy = font.size(text)
    ren = font.render(text, 0, (0, 0, 0))
    pygame.draw.polygon(surf, (255, 255, 255),
        [(x,y), (x+sx, y), (x+sx, y+sy), (x, y+sy)])
    surf.blit(ren, (x, y))
    return sx, sy

def renderTextLines(surf, x, y, lines):
    for line in lines:
        y += renderText(surf, x, y, line)[1]

class KeepAwayFromWalls:
    def clipVelocityToWalls(self, sx, sy, dx, dy, W, H):
        # clip the velocity vector to the wall
        map = self.map.map
        for x,y in (sx, sy), (sx+W, sy), (sx+W, sy+H), (sx, sy+H):
            my = int(y/32)
            mx = int(x/32)
            if dx > 0:
                ndx = 32 - x%32
                k = (mx+1, my)
                if map.has_key(k) and map[k].isWall() and dx > ndx:
                    dx = ndx - 1
            elif dx < 0:
                ndx = x%32
                k = (mx-1, my)
                if map.has_key(k) and map[k].isWall() and abs(dx) > ndx:
                    dx = -(ndx - 1)
            if dy > 0:
                ndy = 32 - y%32
                k = (mx, my+1)
                if map.has_key(k) and map[k].isWall() and dy > ndy:
                    dy = ndy - 1
            elif dy < 0:
                ndy = y%32
                k = (mx, my-1)
                if map.has_key(k) and map[k].isWall() and abs(dy) > ndy:
                    dy = -(ndy - 1)
        return dx, dy

opj = os.path.join
class Player(KeepAwayFromWalls):
    stand_image = pygame.image.load(opj('data', 'player-standing.png'))
    face_down_image = stand_image
    face_up_image = pygame.transform.flip(stand_image, 0, 1)
    face_left_image = pygame.transform.rotate(stand_image, -90)
    face_right_image = pygame.transform.rotate(stand_image, 90)
    fixer_image = pygame.image.load(opj('data', 'fixer.png'))
    fixer_down_image = fixer_image
    fixer_up_image = pygame.transform.flip(fixer_image, 0, 1)
    fixer_left_image = pygame.transform.rotate(fixer_image, -90)
    fixer_right_image = pygame.transform.rotate(fixer_image, 90)
    boom_image = pygame.image.load(opj('data', 'boom.png'))

    # images for walking
    middle_image = pygame.image.load(opj('data', 'player-middle.png'))
    left_image = pygame.image.load(opj('data', 'player-left-front.png'))
    right_image = pygame.image.load(opj('data', 'player-right-front.png'))
    v_walk_images = [right_image, middle_image, left_image, middle_image]
    h_walk_images = [
        pygame.transform.rotate(right_image, 90),
        pygame.transform.rotate(middle_image, 90),
        pygame.transform.rotate(left_image, 90),
        pygame.transform.rotate(middle_image, 90),
    ]

    WIDTH=26
    HEIGHT=17

    STAND = 0
    MOVE = 1
    FIX = 2
    THROW = 3
    EXPLODE = 4

    def __init__(self, pos, paddock):
        self.pos = pos
        self.paddock = weakref.proxy(paddock)
        self.map = paddock.map
        self.velocity = 0, 0

        # action dir
        self.facing = 'up'

        # action?
        self.action = self.STAND

        # index into walk/fix/throw images
        self.image_index = None

        # a list of keys that have been pressed turning me around
        self.face_keys = []

    def animate(self):
        sx, sy = self.pos
        dx = dy = 0

        # if we're not already exploding, see if we should be ;)
        if self.action != self.EXPLODE:
            mx = int((sx+16)/32)
            my = int((sy+16)/32)
            if self.map.map[mx, my].isMine():
                self.action = self.EXPLODE
                self.explode_timer = 10

        if self.action == self.MOVE:
            # simple movement
            if self.facing == 'up':
                dy = -5
                w = self.WIDTH
                h = self.HEIGHT
            if self.facing == 'down':
                dy = 5
                w = self.WIDTH
                h = self.HEIGHT
            if self.facing == 'left':
                dx = -5
                h = self.WIDTH
                w = self.HEIGHT
            if self.facing == 'right':
                dx = 5
                h = self.WIDTH
                w = self.HEIGHT
            dx, dy = self.clipVelocityToWalls(sx, sy, dx, dy, w, h)

            # walking frame image
            if self.image_index is None:
                self.image_index = 0
            else:
                self.image_index = (self.image_index + 1)%8

        elif self.action == self.FIX:
            # find the wall in the fix direction
            gx, gy = int((sx+16)/32), int((sy+16)/32)
            map = self.paddock.map
            fixing = None
            if self.facing == 'up' and map.map[gx, gy-1].isRubble():
                fixing = map.map[gx, gy-1]
            if self.facing == 'down' and map.map[gx, gy+1].isRubble():
                fixing = map.map[gx, gy+1]
            if self.facing == 'left' and map.map[gx-1, gy].isRubble():
                fixing = map.map[gx-1, gy]
            if self.facing == 'right' and map.map[gx+1, gy].isRubble():
                fixing = map.map[gx+1, gy]
            if fixing:
                self.paddock.map.fixWall(fixing.grid)

        elif self.action == self.THROW:
            # find the closest sheep to me in the throw direction
            possible = []
            pi = math.pi
            for sheep in self.paddock.sheep:
                shx, shy = sheep.pos
                dx = sx - shx
                dy = sy - shy

                # too far away?
                m = magnitude(dx, dy)
                if m > 64:
                    continue

                # check direction and add to possibles if ok
                angle = math.atan2(dx, dy)
                if -pi/4 < angle < pi/4 and self.facing == 'up':
                    possible.append((m, sheep, (dx, dy)))
                elif -3*pi/4 < angle < -pi/4 and self.facing == 'right':
                    possible.append((m, sheep, (dx, dy)))
                elif pi/4 < angle < 3*pi/4 and self.facing == 'left':
                    possible.append((m, sheep, (dx, dy)))
                elif angle > 3*pi/4 or angle < -3*pi/4 and self.facing == 'down':
                    possible.append((m, sheep, (dx, dy)))
                else:
                    continue

            if possible:
                # pick the closest sheep and throw it
                possible.sort()
                self.target = possible[0][1]
                dx, dy = normalise(*possible[0][2])
                self.target.throw(dx*64, dy*64)

            self.action = self.STAND

            dx = dy = 0

        elif self.action == self.EXPLODE:
            # time out the boom graphic
            self.explode_timer -= 1
            if not self.explode_timer:
                self.paddock.gameOver()

        self.pos = (sx+dx, sy+dy)

    def draw(self, surf):
        if self.action == self.STAND:
            if self.facing == 'up':
                surf.blit(self.face_up_image, self.pos)
            elif self.facing == 'down':
                surf.blit(self.face_down_image, self.pos)
            elif self.facing == 'left':
                surf.blit(self.face_left_image, self.pos)
            elif self.facing == 'right':
                surf.blit(self.face_right_image, self.pos)
        elif self.action == self.MOVE:
            if self.facing in ('up', 'down'):
                surf.blit(self.v_walk_images[self.image_index/2], self.pos)
            else:
                surf.blit(self.h_walk_images[self.image_index/2], self.pos)
        elif self.action == self.THROW:
            surf.blit(self.image, self.pos)
        elif self.action == self.FIX:
            if self.facing == 'up':
                surf.blit(self.fixer_up_image, self.pos)
            elif self.facing == 'down':
                surf.blit(self.fixer_down_image, self.pos)
            elif self.facing == 'left':
                surf.blit(self.fixer_left_image, self.pos)
            elif self.facing == 'right':
                surf.blit(self.fixer_right_image, self.pos)
        elif self.action == self.EXPLODE:
            x, y = self.pos
            x += self.WIDTH/2
            y += self.HEIGHT/2
            x -= 128
            y -= 64
            surf.blit(self.boom_image, (x,y))

    keys = {K_UP:'up', K_DOWN:'down', K_LEFT:'left', K_RIGHT:'right'}
    def move(self, key):
        self.image_index = None
        self.action = self.MOVE
        self.facing = self.keys[key]
        self.face_keys.append(self.facing)

    def stop_move(self, key):
        self.image_index = None
        facing = self.keys[key]
        self.face_keys.remove(facing)
        if self.face_keys:
            self.facing = self.face_keys[-1]
        elif self.action == self.MOVE:
            self.action = self.STAND

    def fixer(self, onoff):
        self.image_index = None
        self.action = onoff and self.FIX or self.STAND

    def throw(self, onoff):
        self.image_index = None
        self.action = onoff and self.THROW or self.STAND


class Sheep(KeepAwayFromWalls):
    sheep_image = pygame.image.load(os.path.join('data', 'sheep.png'))
    face_left_image = sheep_image
    face_right_image = pygame.transform.flip(sheep_image, 1, 0)
    face_up_image = pygame.transform.rotate(sheep_image, -90)
    face_down_image = pygame.transform.rotate(sheep_image, 90)

    evil_image = pygame.image.load(os.path.join('data', 'sheep_evil.png'))
    evil_left_image = evil_image
    evil_right_image = pygame.transform.flip(evil_image, 1, 0)
    evil_up_image = pygame.transform.rotate(evil_image, -90)
    evil_down_image = pygame.transform.rotate(evil_image, 90)

    boom_image = pygame.image.load(os.path.join('data', 'boom.png'))
    dead_image = pygame.image.load(os.path.join('data', 'sheep_dead.png'))

    SHEEP = 0
    LONER = 1
    EVIL = 2
    EXPLODE = 3
    DEAD = 4
    FLYING = 5
    ESCAPE = 6

    WIDTH = 25
    HEIGHT = 25

    def __init__(self, pos, paddock, difficulty):
        S, L, E, B = self.SHEEP, self.LONER, self.EVIL, self.ESCAPE
        d = difficulty
        self.typelist = [S, S, S, L, L] + [B]*d + [E]*d
        self.pos = pos
        self.velocity = 0, 0
        self.acceleration = 1
        self.max_velocity = 2
        self.type = self.SHEEP
        self.facing = random.choice(['up', 'down', 'left', 'right'])

        self.paddock = weakref.proxy(paddock)

        # convenience
        self.map = paddock.map

        self.changeType()

    def changeType(self):
        # see what sort of sheep this one is, and how long they'll be that
        # sort of sheep for
        S, L, E, X, D, F, B = self.SHEEP, self.LONER, self.EVIL, \
            self.EXPLODE, self.DEAD, self.FLYING, self.ESCAPE
        if self.type in (S, L, E, F):
            self.type = random.choice(self.typelist)
        elif self.type == X:
            self.type = D
            return

        # loners and black sheep need a target
        if self.type == L:
            self.target = random.choice(self.map.field)
            self.check_time = random.randint(100, 400)
        elif self.type == E:
            dest = random.choice(self.map.walldests)
            self.target = dest.pos
            self.walltarget = random.choice(dest.walls)
            self.check_time = random.randint(100, 400)
        elif self.type == B:
            l = [x for x in self.map.walls if x.isRubble()]
            if l:
                self.walltarget = random.choice(l)
                self.target = self.walltarget.pos
                self.check_time = random.randint(100, 400)
            else:
                self.type = S
                self.check_time = random.randint(100, 400)
        elif self.type == S:
            self.check_time = random.randint(100, 400)

    def throw(self, dx, dy):
        ''' Throw the sheep along the vector dx, dy - reset it to being a
            normal sheep too.
        '''
        if self.type in (self.EXPLODE, self.DEAD):
            # exploding or dead sheep can't fly
            return

        # but you can win with government cheese
        self.type = self.FLYING
        self.check_time = 20
        m = magnitude(dx, dy)
        per = m/20
        nx, ny = normalise(dx, dy)
        self.fly_vec = (nx*per, ny*per)

    def animate(self, flock, attracts, repels, others):
        ''' Flock towards the attact, away from repel and try not to get
            too close to others.
        '''
        # nothing to do if we're dead
        if self.type == self.DEAD:
            return

        # see if our type flag should be re-generated
        self.check_time -= 1
        if not self.check_time:
            self.changeType()

        # nothing to do if we're exploding (or we might have just been made
        # dead)
        if self.type in (self.DEAD, self.EXPLODE):
            return

        # fly, my pretties
        if self.type == self.FLYING:
            sx, sy = self.pos
            fx, fy = self.fly_vec
            self.pos = (sx + fx, sy + fy)
            return

        # our current position and velocity
        sx, sy = self.pos
        vx, vy = self.velocity

        # map position...
        mx = int((sx+16)/32)
        my = int((sy+16)/32)
        if self.map.map[mx, my].isMine():
            self.type = self.EXPLODE
            self.check_time = 10
            return

        # figure out the desired direction to go
        dx, dy = 0, 0
        if self.type in (self.LONER, self.EVIL, self.ESCAPE):
            x, y = self.target
            ddx, ddy = x - sx, y - sy
            m = magnitude(ddx, ddy)
            if self.type in (self.EVIL, self.ESCAPE) and m < 32:
                self.map.damageWall(self.walltarget.grid)
                if self.walltarget.isField():
                    # now make me a loner who just wants to get to the
                    # other side
                    other = self.walltarget.otherside
                    if other.isWall():
                        self.check_time = random.randint(100, 400)
                        self.walltarget = self.walltarget.otherside
                    else:
                        self.type = self.LONER
                        self.check_time = random.randint(100, 400)
                        self.target = self.walltarget.otherside.pos
            dx, dy = dx + ddx, dy + ddy
        else:
            x, y = flock
            self.target = (x, y)
            ddx, ddy = x - sx, y - sy
            m = magnitude(ddx, ddy)
            # if the sheep is close, make it less desirable
            if 256 < m < 128:
                m = (m-128)/2
                ddx, ddy = normalise(ddx, ddy)
                ddx *= m
                ddy *= m
            elif m < 128:
                ddx, ddy = 0, 0

            dx, dy = dx + ddx, dy + ddy

        # now other attractions
        for attract in attracts:
            x, y = attract.pos
            ddx, ddy = x - sx, y - sy
            m = magnitude(ddx, ddy)
            dx, dy = dx + ddx, dy + ddy

        # nasty things (just the player at the moment ;)
        for repel in repels:
            # evil sheep ignore
            if self.type == self.EVIL:
                pass

            # repellent force gets stronger the closer it is
            if hasattr(repel, 'pos'):
                x, y = repel.pos
            else:
                x, y = repel
            ddx, ddy = x - sx, y - sy
            m = magnitude(ddx, ddy)
            if m > 128:
                ddx = ddy = 0
            dx, dy = dx - ddx, dy - ddy

        # add in a negative vector for any other sheep too close
        for other in others:
            # ignore exploded sheep ;)
            if other.type in (self.EXPLODE, self.DEAD):
                pass
            ox, oy = other.pos
            odx = ox-sx
            ody = oy-sy
            if magnitude(odx, ody) < 32:
                dx = dx - odx
                dy = dy - ody

        # max out the velocity to the Sheep Top Speed
        if magnitude(dx, dy) > self.max_velocity:
            # cap it
            ndx, ndy = normalise(dx, dy)
            dx = ndx*self.max_velocity
            dy = ndy*self.max_velocity

        # ok, now see what the difference is between that vector and our
        # current velocity vector
        ddx = dx-vx
        ddy = dy-vy
        if magnitude(ddx, ddy) > self.acceleration:
            # cap the acceleration
            nddx, nddy = normalise(ddx, ddy)
            ddx = nddx*self.acceleration
            ddy = nddy*self.acceleration
            dx = vx + ddx
            dy = vy + ddy

        # figure which direction the sheep is facing
        angle = math.atan2(dx, dy)
        pi = math.pi
        if -pi/4 < angle < pi/4:
            self.facing = 'down'
            h = self.WIDTH
            w = self.HEIGHT
        elif -3*pi/4 < angle < -pi/4:
            self.facing = 'left'
            w = self.WIDTH
            h = self.HEIGHT
        elif pi/4 < angle < 3*pi/4:
            self.facing = 'right'
            w = self.WIDTH
            h = self.HEIGHT
        else:
            self.facing = 'up'
            h = self.WIDTH
            w = self.HEIGHT

        # clip the velocity vector to the wall
        dx, dy = self.clipVelocityToWalls(sx, sy, dx, dy, w, h)

        # update velocity and position
        self.velocity = (dx, dy)
        self.pos = (sx + dx, sy + dy)

    def draw(self, surf):
        if self.type == self.EVIL:
            if self.facing == 'up':
                surf.blit(self.evil_up_image, self.pos)
            elif self.facing == 'down':
                surf.blit(self.evil_down_image, self.pos)
            elif self.facing == 'left':
                surf.blit(self.evil_left_image, self.pos)
            elif self.facing == 'right':
                surf.blit(self.evil_right_image, self.pos)
            colour = (255, 0, 0)
        elif self.type == self.DEAD:
            surf.blit(self.dead_image, self.pos)
            colour = None
        elif self.type == self.EXPLODE:
            x, y = self.pos
            x += self.WIDTH/2
            y += self.HEIGHT/2
            x -= 128
            y -= 64
            surf.blit(self.boom_image, (x,y))
            colour = None
        else:
            if self.facing == 'up':
                surf.blit(self.face_up_image, self.pos)
            elif self.facing == 'down':
                surf.blit(self.face_down_image, self.pos)
            elif self.facing == 'left':
                surf.blit(self.face_left_image, self.pos)
            elif self.facing == 'right':
                surf.blit(self.face_right_image, self.pos)
            if self.type == self.LONER:
                colour = (0, 255, 255)
            else:
                colour = (255, 0, 255)

        if self.paddock.debug and colour is not None:
            # debug target lines
            p = tuple([x+16 for x in self.pos])
            t = tuple([x+16 for x in self.target])
            pygame.draw.line(surf, colour, p, t)

        if self.paddock.debug and self.type != self.DEAD:
            x, y = self.pos
            renderText(surf, x, y, '%d:%d'%(self.type,
            self.check_time))

class MapCell:
    h_image = pygame.image.load(os.path.join('data', 'wall_h.png'))
    v_image = pygame.image.load(os.path.join('data', 'wall_v.png'))
    tr_image = pygame.image.load(os.path.join('data', 'wall_tr.png'))
    tl_image = pygame.image.load(os.path.join('data', 'wall_tl.png'))
    br_image = pygame.image.load(os.path.join('data', 'wall_br.png'))
    bl_image = pygame.image.load(os.path.join('data', 'wall_bl.png'))
    field = pygame.image.load(os.path.join('data', 'field.png'))
    minefield = pygame.image.load(os.path.join('data', 'minefield.png'))
    damage = pygame.image.load(os.path.join('data', 'damage.png'))
    destroyed_wall = pygame.image.load(os.path.join('data', 'destroyed_wall.png'))
    def __init__(self, x, y, type):
        self.grid = x, y
        self.pos = x*32, y*32
        self.type = type
        self.old_type = []
        self.health = 50

    def draw(self, surf):
        if self.type == 'h':
            surf.blit(self.h_image, self.pos)
        elif self.type == 'v':
            surf.blit(self.v_image, self.pos)
        elif self.type == 'Q':
            surf.blit(self.tl_image, self.pos)
        elif self.type == 'W':
            surf.blit(self.tr_image, self.pos)
        elif self.type == 'E':
            surf.blit(self.bl_image, self.pos)
        elif self.type == 'R':
            surf.blit(self.br_image, self.pos)
        elif self.type == '*':
            surf.blit(self.minefield, self.pos)
        elif self.type == 'x':
            surf.blit(self.destroyed_wall, self.pos)
        else:
            surf.blit(self.field, self.pos)

    def changeType(self, type):
        self.old_type.append(self.type)
        self.type = type

    def revertType(self):
        self.type = self.old_type.pop()

    def isWall(self):
        return self.type in 'vhQWER'

    def isMine(self):
        return self.type == '*'

    def isField(self):
        return self.type in 'x.'

    def isRubble(self):
        return self.type == 'x'

    def drawDamage(self, surf):
        surf.blit(self.damage, self.pos)

class WallDestSpots:
    ''' Places sheep can go to destroy walls ;)
    '''
    def __init__(self, pos):
        self.pos = pos
        self.walls = []

    def addwall(self, wall):
        self.walls.append(wall)

class Map:
    def __init__(self, map):
        self.cells = []
        self.field = []
        self.walls = []
        self.walldests = []
        self.map = {}
        self.updated = []

        # parse the map string
        map = ''.join(map.split())
        for i in range(320):
            x = (i%20)
            y = (i/20)

            cell = MapCell(x, y, map[i])
            self.cells.append(cell)
            self.map[x,y] = cell
            if cell.isWall():
                self.walls.append(cell)
            if cell.isField():
                self.field.append((x*32, y*32))

        # find all the field cells with walls around 'em
        d = {}
        sd = d.setdefault
        for field in self.field:
            x, y = field
            x /= 32
            y /= 32
            if self.map.has_key((x-1, y)) and self.map[x-1,y].isWall():
                wall = self.map[x-1,y]
                wall.otherside = self.map[x-2,y]
                sd(field, WallDestSpots(field)).addwall(wall)
            if self.map.has_key((x+1, y)) and self.map[x+1,y].isWall():
                wall = self.map[x+1,y]
                wall.otherside = self.map[x+2,y]
                sd(field, WallDestSpots(field)).addwall(wall)
            if self.map.has_key((x, y-1)) and self.map[x,y-1].isWall():
                wall = self.map[x,y-1]
                wall.otherside = self.map[x,y-2]
                sd(field, WallDestSpots(field)).addwall(wall)
            if self.map.has_key((x, y+1)) and self.map[x,y+1].isWall():
                wall = self.map[x,y+1]
                wall.otherside = self.map[x,y+2]
                sd(field, WallDestSpots(field)).addwall(wall)
        self.walldests = d.values()

    def damageWall(self, grid):
        wall = self.map[grid]
        wall.health -= 1
        if wall.health <= 0:
            # wall destroyed, so change it into field
            wall.health = 0
            self.changeType(grid, 'x')
# XXX
#        else:
#            self.walltarget.drawDamage(surf)

    def fixWall(self, grid):
        wall = self.map[grid]

        # fix wall incrementally needs animation...
#        wall.health += 10
        wall.health = 50
        if wall not in self.updated:
            self.updated.append(wall)
        if wall.health >= 50:
            # wall repaired
            wall.health = 50
            self.revertType(grid)
# XXX
#        else:
#            self.walltarget.drawFixing(surf)

    def changeType(self, grid, type):
        cell = self.map[grid]
        cell.changeType(type)
        if cell not in self.updated:
            self.updated.append(cell)

    def revertType(self, grid):
        cell = self.map[grid]
        cell.revertType()
        if cell not in self.updated:
            self.updated.append(cell)

    def refresh(self, surf):
        for section in self.updated:
            section.draw(surf)
        self.updated = []

    def draw(self, surf):
        for section in self.cells:
            section.draw(surf)
            if (section.isWall() or section.isRubble()) and section.health < 50:
                renderText(surf, section.x, section.y, int(section.health))

class Game:
    NUM_SHEEP = 20

    from maps import *

    GAME_OVER = 0
    RUNNING = 1
    LEVEL_PRELUDE = 2
    PAUSED = 3
    HELP = 4
    SPLASH_SCREEN = 5
    LEVEL_FINISHED = 6
    GAME_FINISHED = 7

    # 60 seconds
    #GAME_LENGTH = 1*1000
    GAME_LENGTH = 30*1000

    def __init__(self, size=(640, 512)):
        self.windowSize = size
        self.surf = pygame.display.set_mode(self.windowSize,
            DOUBLEBUF|HWSURFACE)
        pygame.display.set_caption('Baaaaaaaad sheep ')
        self.field = pygame.Surface(self.surf.get_size(),
            self.surf.get_flags(), self.surf)
        self.ticks = pygame.time.get_ticks()

        # some colours to work with
        self.white = self.surf.map_rgb(255, 255, 255)
        self.ground = self.surf.map_rgb(0x76, 0xad, 0x6b)
        self.black = self.surf.map_rgb(0, 0, 0)

        # show debug info? flag
        self.debug = 0

        # game state...
        self.level = 1
        self.map_source = self.basic
        self.num_sheep = self.NUM_SHEEP
        self.state = self.SPLASH_SCREEN
        self.state_ticks = None
        #self.state = self.RUNNING
        #self.time_length = self.GAME_LENGTH
        #self.state_ticks = self.time_length
        self.old_state = None

        self.init()

    def init(self):
        # wall is stored as blocks in the source string
        self.map = Map(self.map_source.strip()) 
        self.map.draw(self.field)
        self.flock = random.choice(self.map.field)
        # add some sheep and the player!
        self.populate()

    def populate(self):
        self.sheep = []
        taken = {}
        start = random.choice(self.map.field)
        for i in range(self.num_sheep):
            while taken.has_key(start):
                start = random.choice(self.map.field)
            self.sheep.append(Sheep(start, self, self.level))
            taken[start] = 1
        while taken.has_key(start):
            start = random.choice(self.map.field)
        self.player = Player(start, self)

    def animateSheep(self):
        # flock to an open field position
        flock = self.flock
        # XXX time out the flock pos to make them all move

        # figure the average center of the sheep
#        num = len(self.sheep)
#        totx, toty = 0, 0
#        for sheep in self.sheep:
#            x, y  = sheep.pos
#            totx += x
#            toty += y
#        flock = (totx/num, toty/num), 

        # now move the sheep around
        n = 0
        for sheep in self.sheep:
            if sheep.type == sheep.DEAD:
                n += 1
            sheep.animate(flock, [], [self.player], self.sheep)
        if n == len(self.sheep):
            self.gameOver()


    def drawSheep(self):
        for sheep in self.sheep:
            sheep.draw(self.surf)

    def renderFPS(self):
        now = pygame.time.get_ticks()
        diff = now - self.ticks
        text = "fps: %.2g"%(1000. / diff)
        self.ticks = now
        if self.state_ticks and self.state == self.RUNNING:
            self.state_ticks -= diff
        if self.debug:
            renderText(self.surf, 0, 0, text)

    def renderTimer(self):
        pygame.draw.polygon(self.surf, (0,0,0), [(270, 2), (370, 2),
            (370, 12), (270,12)])
        l = self.state_ticks*100./self.time_length
        pygame.draw.polygon(self.surf, (255,0,0), [(270, 2), (270+l, 2),
            (270+l, 12), (270,12)])

    def renderHelp(self):
        renderTextLines(self.surf, 50, 100, ['HELP', 'arrows: move around',
            'SHIFT+arrow: repair walls', 'CTRL+arrow: throw sheep',
            'd: show debugging info', 'p: pause',
            ' ',
            'h: close the help'])

    def renderGameOver(self):
        renderTextLines(self.surf, 50, 100, ['GAME OVER',
            'press space for a new game', 'press h for help'])

    def renderSplash(self):
        renderTextLines(self.surf, 50, 100, ['WELCOME',
            'Save the baaaaaahhhd sheep',
            ' ',
            'The grass is greener on the outside.',
            'Unfortunately, it\'s mined out there.',
            'Sheep and mines don\'t mix.',
            'Keep your flock alive!',
            ' ',
            'Save the sheep by fixing the walls they wreck.',
            'You can also throw sheep over your shoulder.',
            ' ',
            'press <space> to start',
            ' ',
            'press h for help'])

    def renderFinished(self):
        renderTextLines(self.surf, 50, 100, [
            'LEVEL %d passed!'%(self.level-1),
            ' ',
            'You have advanced to the next level',
            ' ',
            'press <space> to start',
            ' ',
            'press h for help'])

    def renderAllFinished(self):
        renderTextLines(self.surf, 50, 100, [
            'Congratulations, you finished the game!',
            ' ',
            'You saved %d sheep.'%self.sheep_saved,
            ' ',
            'press <space> to restart',
            ' ',
            'press h for help'])

    def render(self):
        if self.state in (self.RUNNING, self.PAUSED):
            self.map.refresh(self.field)
            self.surf.blit(self.field, (0,0))
            if self.state == self.RUNNING:
                self.animateSheep()
                self.player.animate()
            self.drawSheep()
            self.player.draw(self.surf)
            if self.state == self.PAUSED:
                renderText(self.surf, 325, 256, 'PAUSED (h for help)')
            self.renderTimer()
        elif self.state == self.HELP:
            self.surf.fill((255, 255, 255))
            self.renderHelp()
        elif self.state == self.GAME_OVER:
            self.surf.fill((255, 255, 255))
            self.renderGameOver()
        elif self.state == self.SPLASH_SCREEN:
            self.surf.fill((255, 255, 255))
            self.renderSplash()
        elif self.state == self.LEVEL_FINISHED:
            self.surf.fill((255, 255, 255))
            self.renderFinished()
        elif self.state == self.GAME_FINISHED:
            self.surf.fill((255, 255, 255))
            self.renderAllFinished()
        self.renderFPS()

    def gameOver(self):
        self.state = self.GAME_OVER

    def run(self):
        # use timer to limit speed
        SHEEPTICK = 25
#        pygame.time.set_timer(SHEEPTICK, 1000/10)
        while 1:
            if self.state_ticks is not None and self.state_ticks < 0:
                self.level += 1
                self.state = self.LEVEL_FINISHED
                self.state_ticks = None
                if self.level == 16:
                    self.state = self.GAME_FINISHED
                elif self.level >= 12:
                    self.map_source = self.horseshoe
                    self.state_ticks = self.GAME_LENGTH
                elif self.level >= 8:
                    self.map_source = self.heich
                    self.state_ticks = self.GAME_LENGTH
                elif self.level >= 4:
                    self.map_source = self.island
                    self.state_ticks = self.GAME_LENGTH
                elif self.level >= 1:
                    self.map_source = self.basic
                    self.state_ticks = self.GAME_LENGTH

                # total count of sheep saved in whole game
                num = len([s for s in self.sheep if s.type != s.DEAD])
                self.sheep_saved += num

            # draw the game
            self.render()
            pygame.display.flip()

            for e in pygame.event.get():
                if e.type == KEYDOWN:
                    if e.key in (K_ESCAPE, K_q):
                        return
                    elif e.key in (K_UP, K_DOWN, K_LEFT, K_RIGHT):
                        self.player.move(e.key)
                    elif e.key in (K_LSHIFT, K_RSHIFT):
                        self.player.fixer(1)
                    elif e.key in (K_LCTRL, K_RCTRL):
                        self.player.throw(1)
                    elif e.key == K_SPACE and self.state in (self.GAME_OVER,
                            self.LEVEL_FINISHED, self.SPLASH_SCREEN,
                            self.GAME_FINISHED):
                        if self.state in (self.GAME_OVER,
                                self.SPLASH_SCREEN, self.GAME_FINISHED):
                            self.level = 1
                            self.map_source = self.basic
                            self.sheep_saved = 0
                        self.state = self.RUNNING
                        self.time_length = self.GAME_LENGTH
                        self.state_ticks = self.time_length
                        self.init()
                    elif e.key == K_d:
                        self.debug = not self.debug
                    elif e.key == K_p:
                        if self.state == self.PAUSED:
                            self.state = self.RUNNING
                        elif self.state == self.RUNNING:
                            self.state = self.PAUSED
                    elif e.key == K_h:
                        if self.state == self.HELP:
                            self.state = self.old_state
                        else:
                            self.old_state = self.state
                            self.state = self.HELP
                elif e.type == KEYUP:
                    if e.key in (K_UP, K_DOWN, K_LEFT, K_RIGHT):
                        self.player.stop_move(e.key)
                    elif e.key in (K_LSHIFT, K_RSHIFT):
                        self.player.fixer(0)
                    elif e.key in (K_LCTRL, K_RCTRL):
                        self.player.throw(0)
#                elif e.type == SHEEPTICK:
#                    self.render()
#                    pygame.display.flip()
                elif e.type == QUIT:
                    return

if __name__ == '__main__':
    game = Game()
    game.run()


