# Note: using Pygame OpenGL rotating cube example as starting point...

import math
import random
import pygame
from pygame.locals import *


### User options begin. #############################################

#(SCREEN_WIDTH, SCREEN_HEIGHT)  = (1024, 768)
#(SCREEN_WIDTH, SCREEN_HEIGHT)  = (800, 600)
(SCREEN_WIDTH, SCREEN_HEIGHT)  = (640, 480)

RUN_FULL_SCREEN = 0
#RUN_FULL_SCREEN = 1

ZOOM_IN_KEY  = K_e
ZOOM_OUT_KEY = K_q
PAN_KEY      = K_TAB

NUM_STARS  = (2000, 300, 30)

### User options end. #############################################

STAR_SIZES = (1, 2, 4)

# Rate of spin of solar collectors.
INACTIVE_SPIN = 0.001

# Rate at which a shell becomes selected.
SELECT_PHASE_SPEED = 0.1

# Frequency at which a shell blinks.
SELECT_PULSE_FREQ = 1.0 / 30.0 * 2.0 * math.pi

# The height of an orbital separator.
SEPARATOR_HEIGHT = 2

# The base velocity of a shell.
BASE_VEL = 2.0 * math.pi / 200.0

# The base velocity of a rock
ROCK_SPEED = 1

DEAD_ENERGY = 150
ENERGY_COSTS = (300, 200, 100, 75, 60)
CHARGE_SPEED = 0.01
INITIAL_ENERGY = DEAD_ENERGY
MAX_ENERGY = 440
BAR_WIDTH = 50
BAR_PADDING = 20

BEVEL_WEIGHTS = (0.5, 0.75, 0.85, 0.95, 1.0)

INTERFACE_WIDTH  = 640
INTERFACE_HEIGHT = 480

SPACE_WIDTH  = INTERFACE_WIDTH  * 12
SPACE_HEIGHT = INTERFACE_HEIGHT * 12

SCALES = [0.134, 0.167, 0.209, 0.262, 0.327,
          0.409, 0.512, 0.64,  0.8,   1.0,
          1.25,  1.55,  1.95,  2.4,   3.0,
          3.8,   4.7,   6.0,   7.45,  9.0,
          10.6, 12.0]

try:
    from OpenGL.GL import *
    from OpenGL.GLUT import *
except:
    print "Orbital Defense requires PyOpenGL (http://pyopengl.sf.net)"
    raise SystemExit


def main():
    game = Game()
    while game.run():
        pass

class Game:
    STATE_RUNNING = 0
    STATE_GAMEOVER = 1

    STATE_DEFAULT = 0
    STATE_PANNING = 1
    STATE_ARCING  = 2

    def __init__(self):
        # Initialize screen mode.
        pygame.init()
        flags = OPENGL | DOUBLEBUF
        if RUN_FULL_SCREEN:
            flags |= FULLSCREEN
        pygame.display.set_mode((SCREEN_WIDTH,SCREEN_HEIGHT), flags)

        glEnable(GL_DEPTH_TEST)
        glShadeModel(GL_SMOOTH)
        glEnable(GL_LIGHTING)
        glEnable(GL_LIGHT0)
        glEnable(GL_LIGHT1)
        glEnable(GL_TEXTURE_2D)

        glCullFace(GL_BACK)
        glEnable(GL_CULL_FACE)

        #glEnable(GL_LINE_SMOOTH)
        glEnable(GL_POINT_SMOOTH)
        glEnable(GL_BLEND)
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)
        glHint(GL_POINT_SMOOTH_HINT, GL_FASTEST)
        #glHint(GL_LINE_SMOOTH_HINT, GL_DONT_CARE)
        #glHint(GL_LINE_SMOOTH_HINT, GL_NICEST)

        # Set up camera.
        glMatrixMode(GL_PROJECTION)
        glLoadIdentity()
        glViewport(0, 0, SCREEN_WIDTH, SCREEN_HEIGHT)
        glOrtho(   0,  INTERFACE_WIDTH,
                   0, INTERFACE_HEIGHT,
                -100,              100)

        self.background = Background()
        self.earth = Planet(80, 30, 40, "land.png")
        self.shells = []
        self.shells.append(Shell(self, 120, 20, 35,  5, 50, (1, 0.0, 0.0, 0.6)))
        self.shells.append(Shell(self, 155, 15, 35,  7, 50, (1.0, 0.5, 0.0, 0.6)))
        self.shells.append(Shell(self, 190, 10, 30, 11, 40, (0.0, 1.0, 0.0, 0.5)))
        self.shells.append(Shell(self, 220, 10, 30, 17, 30, (0.1, 0.3, 1.0, 0.7)))
        self.shells.append(Shell(self, 270,  3, 30, 23, 10, (0.6, 0.6, 1.0, 1.0), 0))

        self.angle = 0

        self.mx = 0
        self.my = 0
        self.omx = 0
        self.omy = 0
        self.worldX = 0
        self.worldY = 0

        self.panX = 0
        self.panY = 0
        self.scaleIndex = SCALES.index(1.0) - 3
        self.state = self.STATE_DEFAULT
        self.mouseShell   = None
        self.mouseSegment = None
        self.mouseAngle = 0
        self.time = 0
        self.rocks       = []
        self.energy      = INITIAL_ENERGY
        self.particles   = []
        surf = pygame.image.load("game_over.png")
        self.gameOver = pygame.image.tostring(surf, "RGBA")
        self.gameOverWidth  = surf.get_width()
        self.gameOverHeight = surf.get_height()
        self.gameState = self.STATE_GAMEOVER

    def run(self):
        while self.gameState == self.STATE_GAMEOVER:
            startTime = pygame.time.get_ticks()
            event = pygame.event.poll()
            if   event.type == QUIT or (event.type == KEYDOWN and
                                        event.key == K_ESCAPE):
                return 0
            elif event.type == MOUSEBUTTONDOWN:
                self.gameState = self.STATE_RUNNING
            else:
                self.draw()
                self.drawGameOver()
                deltaTime = (pygame.time.get_ticks() - startTime) / 10
                self.angle += deltaTime
                pygame.display.flip()

        self.energy = INITIAL_ENERGY
        newLevel = 1
        level = 0
        while self.gameState == self.STATE_RUNNING:
            if newLevel:
                level += 1
                newLevel = 0
                for i in range(5 * level * level):
                    scatterDist = SPACE_HEIGHT * level
                    self.rocks.append(Rock(self,
                                           random.random() * 2.0 * math.pi,
                                           random.random() * scatterDist +
                                           SPACE_HEIGHT / 2,
                                           random.randrange(5)))

            startTime = pygame.time.get_ticks()
            event = pygame.event.poll()
            if   event.type == QUIT or (event.type == KEYDOWN and
                                        event.key == K_ESCAPE):
                return 0
            elif event.type == MOUSEMOTION:
                self.setMouseCoords(event.pos)
                if   self.state == self.STATE_PANNING:
                    self.pan()
                elif self.state == self.STATE_ARCING:
                    pass
            elif event.type == MOUSEBUTTONDOWN:
                self.setMouseCoords(event.pos)
                if   event.button == 4: # Mouse wheel up.
                    self.zoomIn()
                elif event.button == 5: # Mouse wheel down.
                    self.zoomOut()
                elif event.button == 2: # Middle mouse button.
                    self.state = self.STATE_PANNING
                elif event.button == 1: # Left mouse button.
                    if self.mouseShell is not None:
                        self.mouseShell.setMouseAngle0(self.worldX,
                                                       self.worldY)
                        self.state = self.STATE_ARCING
                elif event.button == 3: # Right mouse button.
                    self.placeShield()
            elif event.type == MOUSEBUTTONUP:
                if   event.button == 2: # Middle mouse button.
                    self.state = self.STATE_DEFAULT
                elif event.button == 1: # Left mouse button.
                    self.state = self.STATE_DEFAULT
            elif event.type == KEYDOWN:
                if   event.key == ZOOM_IN_KEY:
                    self.zoomIn()
                elif event.key == ZOOM_OUT_KEY:
                    self.zoomOut()
                elif event.key == PAN_KEY:
                    self.state = self.STATE_PANNING
            elif event.type == KEYUP:
                if event.key == PAN_KEY:
                    self.state = self.STATE_DEFAULT
            elif event.type == NOEVENT:
                self.draw()
                deltaTime = (pygame.time.get_ticks() - startTime) / 10
                self.process(deltaTime)
                self.angle += deltaTime
                pygame.display.flip()
        return 1

    def setMouseCoords(self, pos):
        scale = SCALES[self.scaleIndex]
        self.omx = self.mx
        self.omy = self.my
        self.mx =   float(pos[0]) * INTERFACE_WIDTH  / SCREEN_WIDTH
        self.my = (-float(pos[1]) * INTERFACE_HEIGHT / SCREEN_HEIGHT +
                    INTERFACE_HEIGHT)
        self.worldX = (self.mx - INTERFACE_WIDTH / 2.0)  / scale - self.panX
        self.worldY = (self.my - INTERFACE_HEIGHT / 2.0) / scale - self.panY

    def process(self, time):
        self.time += time

        if self.state == self.STATE_ARCING:
            self.mouseShell.setMouseAngle1(self.worldX,
                                           self.worldY)
        else:
            # Find the shell that the mouse is on.
            if self.mouseShell is not None:
                self.mouseShell.setSelected(0)
                self.mouseShell = None

            for shell in self.shells:
                if shell.mouseOnShell(self.worldX, self.worldY):
                    shell.setSelected(1, self.worldX, self.worldY)
                    self.mouseShell = shell
                    break

        for shell in self.shells:
            if shell == self.mouseShell:
                shell.process(time)
            else:
                shell.process(time)

        energyShell = self.shells[-1]
        newEnergy = energyShell.getEnergy() * CHARGE_SPEED * time
        self.energy = min(self.energy + newEnergy, MAX_ENERGY)

        deadRocks = []
        for rock in self.rocks:
            rock.process(time)
            if rock.state == rock.STATE_DEAD:
                deadRocks.append(rock)
        for rock in deadRocks:
            self.rocks.remove(rock)

        if len(self.rocks) == 0:
            newLevel = 1

        deadParticles = []
        for particle in self.particles:
            particle.process(time)
            if particle.state == particle.STATE_DEAD:
                deadParticles.append(particle)
        for particle in deadParticles:
            self.particles.remove(particle)

    def draw(self):
        #glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT)
        glClear(GL_COLOR_BUFFER_BIT)
        glMaterialfv(GL_FRONT, GL_SPECULAR, (1, 1, 1, 1))
        glMaterialfv(GL_FRONT, GL_SHININESS, (20.0))
        glLightfv(GL_LIGHT0, GL_DIFFUSE, (2, 2, 2, 1))
        glLightfv(GL_LIGHT1, GL_DIFFUSE, (0.2, 0.2, 0.2, 1))

        # Initialize modeling transformation.
        glMatrixMode(GL_MODELVIEW)
        glLoadIdentity()

        # Put lights in the scene.
        glLightfv(GL_LIGHT0, GL_POSITION, (1, 1, 1, 0))
        glLightfv(GL_LIGHT1, GL_POSITION, (-1, -1, 0, 0))

        glPushMatrix()

        # Center the screen.
        glTranslatef(320, 240, 0)

        scale = SCALES[self.scaleIndex]
        glScalef(scale, scale, 1.0)
        glTranslatef(self.panX, self.panY, 0)
        glDisable(GL_DEPTH_TEST)

        # Draw the background.
        glDisable(GL_LIGHTING)
        self.background.draw(scale)

        # Draw the incoming arcs.
        glLineWidth(2.0)
        for rock in self.rocks:
            rock.drawTrail()
        glEnable(GL_LIGHTING)

        # Draw the Earth.
        glPushMatrix()
        glRotatef(30, 0, 0, 1)
        glRotatef(10, 1, 0, 0)
        glRotatef(-90, 1, 0, 0)
        glRotatef(self.angle, 0, 0, 1)
        self.earth.draw()
        glPopMatrix()

        # Draw the shells.
        glDisable(GL_LIGHTING)
        glLineWidth(2.0)
        for shell in self.shells:
            shell.draw()

        glLineWidth(1)
        glPopMatrix()
        glColor4f(1, 1, 1, 0.5)
        glBegin(GL_QUADS)
        glVertex2f(BAR_PADDING, BAR_PADDING)
        glVertex2f(BAR_WIDTH + BAR_PADDING, BAR_PADDING)
        glVertex2f(BAR_WIDTH + BAR_PADDING, self.energy + BAR_PADDING)
        glVertex2f(BAR_PADDING, self.energy + BAR_PADDING)
        glEnd()

        glBegin(GL_LINE_LOOP)
        glVertex2f(BAR_PADDING + 1, BAR_PADDING + 1)
        glVertex2f(BAR_WIDTH + BAR_PADDING, BAR_PADDING)
        glVertex2f(BAR_WIDTH + BAR_PADDING, MAX_ENERGY + BAR_PADDING - 1)
        glVertex2f(BAR_PADDING + 1, MAX_ENERGY + BAR_PADDING - 1)
        glEnd()

        glLineWidth(2)
        glBegin(GL_LINES)
        glVertex2f(BAR_PADDING, DEAD_ENERGY)
        glVertex2f(BAR_WIDTH + BAR_PADDING, DEAD_ENERGY)
        glEnd()
        glLineWidth(1)

        glBegin(GL_LINES)
        for i in range(len(ENERGY_COSTS)):
            glColor4fv(self.shells[i].color)
            cost = ENERGY_COSTS[i]
            glVertex2f(BAR_PADDING, cost)
            glVertex2f(BAR_WIDTH + BAR_PADDING, cost)
        glEnd()

        glEnable(GL_LIGHTING)

    def drawGameOver(self):
        glPixelStorei(GL_UNPACK_ALIGNMENT, 1)
        glTexImage2D(GL_TEXTURE_2D, 0, 3, self.gameOverWidth, self.gameOverHeight,
            0, GL_RGBA, GL_UNSIGNED_BYTE, self.gameOver)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR)
        glDisable(GL_LIGHTING)
        glBegin(GL_QUADS)
        glTexCoord2f(0, 0.5)
        glVertex2f(320 - 50, 340)
        glTexCoord2f(1, 0.5)
        glVertex2f(320 + 50, 340)
        glTexCoord2f(1, 0)
        glVertex2f(320 + 50, 340 + 50)
        glTexCoord2f(0, 0)
        glVertex2f(320 - 50, 340 + 50)
        glEnd()
        glEnable(GL_LIGHTING)

    def zoomIn(self):
        self.scaleIndex = min(self.scaleIndex + 1, len(SCALES) - 1)

    def zoomOut(self):
        self.scaleIndex = max(self.scaleIndex - 1, 0)

    def pan(self):
        scale = SCALES[self.scaleIndex]
        deltaX = self.mx - self.omx
        deltaY = self.my - self.omy
        self.panX += deltaX / scale
        self.panY += deltaY / scale

        # Clip to visible space (can't lose sight of Earth when zoomed out).
        halfSpaceWidth  = SPACE_WIDTH  / 2.0
        halfSpaceHeight = SPACE_HEIGHT / 2.0
        halfViewWidth  = INTERFACE_WIDTH / scale / 2.0
        halfViewHeight = INTERFACE_WIDTH / scale / 2.0
        safeX1 = -halfSpaceWidth  + halfViewWidth
        safeY1 = -halfSpaceHeight + halfViewHeight
        safeX2 =  halfSpaceWidth  - halfViewWidth
        safeY2 =  halfSpaceHeight - halfViewHeight
        self.panX = max(safeX1, self.panX)
        self.panY = max(safeY1, self.panY)
        self.panX = min(safeX2, self.panX)
        self.panY = min(safeY2, self.panY)

    def placeShield(self):
        if self.mouseShell is None: return
        segment = self.mouseShell.selSegment
        if segment is None: return
        if segment.state != segment.STATE_DEAD: return
        i = self.shells.index(self.mouseShell)

        if self.energy > ENERGY_COSTS[i]:
            segment.state = segment.STATE_ALIVE
            self.energy -= ENERGY_COSTS[i]


class Shell:
    def __init__(self,
                 game,
                 radius,
                 thickness,
                 selThickness,
                 numSegments,
                 segPoints,
                 color,
                 active=1):
        self.game        = game
        self.radius      = radius
        self.thickness   = thickness
        self.selThickness= selThickness
        self.numSegments = numSegments
        self.segPoints   = segPoints
        self.highlight   = (1, 1, 1, 1)
        self.color       = color
        self.active      = active
        self.vel         = 0
        self.rotation    = 0
        self.baseColor   = color
        self.selectPhase = 0.0
        self.selSegment  = None
        self.mouseAngle0 = 0
        self.mouseAngle1 = 0

        self.segments = []
        angleFrac = 2.0 * math.pi / self.numSegments
        for i in range(self.numSegments):
            startAngle = i       * angleFrac
            endAngle   = (i + 1) * angleFrac
            alive = i % 2
            self.segments.append(ShellSegment(self, startAngle, endAngle, alive))

    # Given mouse coordinates relative to the shell, returns None if mouse
    # is not on a handle.  Otherwise, return adjusted point in polar coords.
    def mouseOnShell(self, x, y):
        dist = math.sqrt(x * x + y * y)
        if (dist > self.radius + self.selThickness or
            dist < self.radius):
            return 0
        return 1

    def setSelected(self, selected, x = 0, y = 0):
        if selected:
            angleFrac = 2.0 * math.pi / self.numSegments
            angle = math.atan2(y, x) - self.rotation
            segNum = int(math.floor(clampCircle(angle) / angleFrac))
            self.selSegment = self.segments[segNum]
            self.selSegment.setSelected(1, angle)
        else:
            self.selSegment.setSelected(0)
            self.selSegment = None

    def getGuardianSegment(self, angle, dist1, dist2):
        if (dist2 < self.radius or
            dist1 > self.radius + self.thickness):
            return None
        angle = angle - self.rotation
        angleFrac = 2.0 * math.pi / self.numSegments
        segNum = int(math.floor(clampCircle(angle) / angleFrac))
        segment = self.segments[segNum]
        return segment

    def setMouseAngle0(self, x0, y0):
        if not self.active: return
        self.mouseAngle0 = clampCircle(math.atan2(y0, x0) - self.rotation)
        self.mouseAngle1 = self.mouseAngle0

    def setMouseAngle1(self, x1, y1):
        if not self.active: return
        self.mouseAngle1 = clampCircle(math.atan2(y1, x1) - self.rotation)

    def process(self, time):
        if not self.active:
            self.rotation -= time * INACTIVE_SPIN
            while self.rotation < 0: self.rotation += math.pi * 2.0
        else:
            self.processRotation(time)

        delta = time * SELECT_PHASE_SPEED
        if self.selSegment is not None:
            self.selectPhase = min(1.0, self.selectPhase + delta)
        else:
            self.selectPhase = max(0.0, self.selectPhase - delta)
        pulseAmount = math.cos(self.game.time * SELECT_PULSE_FREQ) / 2.0 + 0.5
        pulseAmount = 1.0 + (0.5 - 1.0) * pulseAmount
        self.highlight = (1, 1, 1, pulseAmount)
        self.color = blendVectors4(self.baseColor,
                                   self.highlight,
                                   self.selectPhase * 0.5)
        for segment in self.segments:
            segment.process(time)

    def processRotation(self, time):
        if self.mouseAngle0 == self.mouseAngle1:
            return

        segsLeft = 0
        for segment in self.segments:
            if segment.state == segment.STATE_ALIVE:
                segsLeft += 1
        mass = self.thickness * segsLeft / self.numSegments
        if mass == 0: mass = 1
        maxVel = BASE_VEL / mass

        delta1 = clampCircleAround0(self.mouseAngle1 - self.mouseAngle0)
        if delta1 > 0:
            self.vel =  maxVel
        else:
            self.vel = -maxVel
        dist = self.vel * time
        if ((delta1 > 0 and dist > delta1) or
            (delta1 < 0 and dist < delta1)):
            self.vel = 0
            self.rotation += delta1
            self.mouseAngle1 = self.mouseAngle0
        else:
            self.rotation = clampCircle(self.rotation + dist)
            self.mouseAngle1 -= dist

    def draw(self):
        glPushMatrix()
        glRotatef(self.rotation * 180.0 / math.pi, 0, 0, 1)
        for i in range(self.numSegments):
            segment = self.segments[i]
            segment.draw()
        self.drawMouseAngle()
        glPopMatrix()

    def getEnergy(self):
        energy = 0
        for segment in self.segments:
            energy += (segment.lightFactor - 0.5)
        return energy * 2.0
 
    def drawMouseAngle(self):
        # Order angles by CCW-ness.
        deltaAngle = self.mouseAngle1 - self.mouseAngle0
        if clampCircle(deltaAngle) > math.pi:
            angle0 = self.mouseAngle1
            angle1 = self.mouseAngle0
        else:
            angle0 = self.mouseAngle0
            angle1 = self.mouseAngle1

        # Calculate number of points on arc.
        angleRange = clampCircle(angle1 - angle0)
        pointDensity = self.numSegments * self.segPoints / (2.0 * math.pi)
        numPoints = int(math.ceil(pointDensity * angleRange))
        if numPoints < 2: return

        innerRadius = self.radius + self.thickness * 0.2
        outerRadius = self.radius + self.thickness * 1.4
        width = outerRadius - innerRadius
        glColor4f(1.0, 1.0, 0.0, 0.5)
        glBegin(GL_QUAD_STRIP)
        for i in range(numPoints):
            angle = (i * angleRange / (numPoints - 1) + angle0)
            if angle0 == self.mouseAngle0:
                r = (i * width / numPoints) + innerRadius
            else:
                r = ((numPoints - i - 1) * width / numPoints) + innerRadius
            cosAngle = math.cos(angle)
            sinAngle = math.sin(angle)
            innerX = cosAngle * r
            innerY = sinAngle * r
            outerX = cosAngle * outerRadius
            outerY = sinAngle * outerRadius
            glVertex2fv((innerX, innerY))
            glVertex2fv((outerX, outerY))
        glEnd()


class ShellSegment:
    STATE_DEAD = 0
    STATE_ALIVE = 1
    def __init__(self, shell, startAngle, endAngle, state = STATE_ALIVE):
        self.shell = shell
        self.startAngle = startAngle
        self.endAngle = endAngle
        self.state = state
        self.list = 0
        self.outlineList = 0
        self.lightFactor = 0.0
        self.mouseAngle = -1
        self.separatorPoints = self.getSeparatorPoints()

        self.weights = [1.0] * self.shell.segPoints
        if self.shell.active:
            for i in range(len(BEVEL_WEIGHTS)):
                self.weights[i] = BEVEL_WEIGHTS[i]
                self.weights[-i - 1] = BEVEL_WEIGHTS[i]
        self.makeLists()

    def setSelected(self, selected, mouseAngle = -1):
        if selected:
            self.mouseAngle = clampCircle(mouseAngle)
        else:
            self.mouseAngle = -1

    def inArc(self, angle):
        if (angle <= self.startAngle and
            angle < self.endAngle):
            return 1
        else:
            return 0

    def makeLists(self):
        points = self.getPoints(-1)

        self.list = glGenLists(1)
        glNewList(self.list, GL_COMPILE)
        self.drawSolid(points)
        glEndList()

        self.outlineList  = glGenLists(1)
        glNewList(self.outlineList, GL_COMPILE)
        self.drawOutline(points)
        glEndList()

    def process(self, time):
        if self.shell.active:
            pass
        else:
            avgAngle = (self.startAngle + self.endAngle) / 2.0
            turnedAngle = avgAngle + self.shell.rotation
            dirX = math.cos(turnedAngle)
            dirY = math.sin(turnedAngle)
            light = dirX * math.sqrt(2) + dirY * math.sqrt(2)
            if light < 0:
                self.lightFactor = 0.5
            else:
                self.lightFactor = light / 2.0 + 0.5

    def draw(self):
        if self.shell.active:
            points = self.getPoints(self.mouseAngle)
            if self.mouseAngle > 0:
                if self.state == self.STATE_ALIVE:
                    glColor4fv(self.shell.color)
                    self.drawSolid(points)

                glColor4fv(self.shell.highlight)
                self.drawOutline(points)
            elif self.state == self.STATE_ALIVE:
                glColor4fv(self.shell.color)
                glCallList(self.list)
        else:
            adjustedLight = (self.shell.color[0] * self.lightFactor,
                             self.shell.color[1] * self.lightFactor,
                             self.shell.color[2] * self.lightFactor,
                             1.0)
            if self.state == self.STATE_ALIVE:
                glColor4fv(adjustedLight)
                glCallList(self.list)
            if self.mouseAngle > 0:
                glColor4fv(self.shell.highlight)
                glCallList(self.outlineList)
        #self.drawSeparator()

    def drawSolid(self, points):
        glBegin(GL_QUAD_STRIP)
        for (innerPos, outerPos) in points:
            glVertex2fv(innerPos)
            glVertex2fv(outerPos)
        glEnd()

    def drawOutline(self, points):
        glBegin(GL_LINE_STRIP)
        for (innerPos, outerPos) in points:
            glVertex2fv(innerPos)
        glEnd()

        glBegin(GL_LINE_STRIP)
        glVertex2fv(points[0][0])
        for (innerPos, outerPos) in points:
            glVertex2fv(outerPos)
        glVertex2fv(points[-1][0])
        glEnd()

    def getPoints(self, handleAngle):
        if handleAngle > 0:
            weights = self.weights[:]
            offset = self.getAngleOffset(handleAngle)
            for i in range(len(BEVEL_WEIGHTS)):
                weights[offset - i] = 1.5
                weights[offset + i] = 1.5
        else:
            weights = self.weights

        angleRange = self.endAngle - self.startAngle
        innerRadius = self.shell.radius
        width = self.shell.thickness
        points = []
        for i in range(self.shell.segPoints):
            angle = (i * angleRange / self.shell.segPoints +
                     self.startAngle)
            cosAngle = math.cos(angle)
            sinAngle = math.sin(angle)
            innerX = cosAngle * innerRadius
            innerY = sinAngle * innerRadius
            outerX = innerX + cosAngle * width * weights[i]
            outerY = innerY + sinAngle * width * weights[i]
            points.append(((innerX, innerY), (outerX, outerY)))
        return points

    def getSeparatorPoints(self):
        angleRange = self.endAngle - self.startAngle
        angleWidth = angleRange / self.shell.segPoints
        angle1 = angleRange - angleWidth + self.startAngle
        angle2 = angleRange + self.startAngle
        cosAngle1 = math.cos(angle1)
        sinAngle1 = math.sin(angle1)
        cosAngle2 = math.cos(angle2)
        sinAngle2 = math.sin(angle2)
        r = self.shell.radius
        p1X = cosAngle1 *  r
        p1Y = sinAngle1 *  r
        p4X = cosAngle2 *  r
        p4Y = sinAngle2 *  r
        p3X = cosAngle2 * (r + SEPARATOR_HEIGHT)
        p3Y = sinAngle2 * (r + SEPARATOR_HEIGHT)
        p2X = cosAngle1 * (r + SEPARATOR_HEIGHT)
        p2Y = sinAngle1 * (r + SEPARATOR_HEIGHT)

        return ((p1X, p1Y), (p2X, p2Y), (p3X, p3Y), (p4X, p4Y))

    def drawSeparator(self):
        glColor3f(0.4, 0.4, 0.4)
        glBegin(GL_QUADS)
        for p in self.separatorPoints:
            glVertex2fv(p)
        glEnd()


    def getAngleOffset(self, angle):
        # Discretize handle angle to one of the segment offsets.
        angleRange = self.endAngle - self.startAngle
        numPoints = self.shell.segPoints
        angleOffset = int((angle - self.startAngle) * numPoints / angleRange)
        angleOffset = max(angleOffset, len(BEVEL_WEIGHTS))
        angleOffset = min(angleOffset, numPoints - len(BEVEL_WEIGHTS) - 1)
        return angleOffset

    def getMouseAngle(self, angle):
        angleRange = self.endAngle - self.startAngle
        angleOffset = self.getAngleOffset(angle)
        numPoints = self.shell.segPoints
        mouseAngle = angleOffset * angleRange / numPoints + self.startAngle
        return mouseAngle


class Background:
    def __init__(self):
        self.starPositions = []
        self.starSizes = []
        halfSpaceWidth  = SPACE_WIDTH  / 2.0
        halfSpaceHeight = SPACE_HEIGHT / 2.0
        for i in range(3):
            size = STAR_SIZES[i]
            numStars = NUM_STARS[i]
            stars = []
            for j in range(numStars):
                stars.append((random.random() * SPACE_WIDTH  - halfSpaceWidth,
                              random.random() * SPACE_HEIGHT - halfSpaceHeight,
                              0))
            self.starPositions.append(stars)
            self.starSizes.append(size)
        self.lists = []
        for i in range(3):
            list = glGenLists(1)
            self.lists.append(list)
            glNewList(list, GL_COMPILE)
            glBegin(GL_POINTS)
            stars = self.starPositions[i]
            for star in stars:
                glVertex3fv(star)
            glEnd()
            glEndList()

    def draw(self, scale):
        for i in range(3):
            size = STAR_SIZES[i] * scale
            if size < 1.0:
                glColor3f(size, size, size)
            else:
                glColor3f(1.0, 1.0, 1.0)
            stars = self.starPositions[i]
            glPointSize(size)
            glCallList(self.lists[i])


class Planet:
    def __init__(self, radius, uSteps, vSteps, skinFile):
        self.radius = radius
        self.uSteps = uSteps
        self.vSteps = vSteps
        surf = pygame.image.load(skinFile)
        self.tex = pygame.image.tostring(surf, "RGBA")
        self.texWidth  = surf.get_width()
        self.texHeight = surf.get_height()
        self.verts = []
        self.norms = []
        self.texCoords = []
        self.tris = []
        self.list = 0
        self.makePlanet()
        self.makeList()

    def draw(self):
        glPixelStorei(GL_UNPACK_ALIGNMENT, 1)
        glTexImage2D(GL_TEXTURE_2D, 0, 3, self.texWidth, self.texHeight,
            0, GL_RGBA, GL_UNSIGNED_BYTE, self.tex)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR)
        glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR)
        #glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL)
        glCallList(self.list)

    def makeList(self):
        self.list = glGenLists(1)
        glNewList(self.list, GL_COMPILE)
        glBegin(GL_TRIANGLES)
        for tri in self.tris:
            for v in tri:
                glTexCoord2fv(self.texCoords[v])
                glNormal3fv(self.norms[v])
                glVertex3fv(self.verts[v])
        glEnd()
        glEndList()

    def makePlanet(self):
        # Make points.
        for i in range(self.uSteps):
            u = float(i) / (self.uSteps - 1)
            phi    = i * math.pi / (self.uSteps - 1)
            z      = math.cos(phi) * self.radius
            zScale = math.sin(phi)
            for j in range(self.vSteps + 1):
                v = float(j) / self.vSteps
                theta = j * math.pi * 2.0 / self.vSteps
                x = math.cos(theta) * zScale * self.radius
                y = math.sin(theta) * zScale * self.radius
                self.verts.append((x, y, z))
                self.norms.append(normalize((x, y, z)))
                self.texCoords.append((v,u))

        # Connect points.
        for i in range(self.uSteps - 1):
            ring1 = []
            ring2 = []
            for j in range(self.vSteps + 1):
                ring1.append(i     * (self.vSteps + 1) + j)
                ring2.append((i+1) * (self.vSteps + 1) + j)
            # Stitch rings together.
            for j in range(self.vSteps):
                p1 = ring1[j]
                p2 = ring1[j+1]
                p3 = ring2[j+1]
                p4 = ring2[j]
                self.tris.append((p1, p3, p2))
                self.tris.append((p1, p4, p3))

class Rock:
    STATE_DEAD  = 0
    STATE_ALIVE = 1
    def __init__(self, game, angle, dist, size):
        self.game  = game
        self.angle = angle
        self.dist  = dist
        self.size  = size
        self.posX  = math.cos(self.angle) * self.dist
        self.posY  = math.sin(self.angle) * self.dist
        self.state = self.STATE_ALIVE

    def drawTrail(self):
        closeness = self.dist / (SPACE_WIDTH / 4.0)
        if closeness < 1.0:
            weight = (1.0 - closeness)
            color = self.game.shells[self.size].color
            glColor3f(color[0] * weight,
                      color[1] * weight,
                      color[2] * weight)
            glBegin(GL_LINES)
            glVertex2f(self.posX, self.posY)
            glVertex2f(0, 0)
            glEnd()

    def draw(self):
        glColor3v(0.5, 0.5, 0.5)
        glPointSize(self.size)
        glBegin(GL_POINTS)
        glVertex2f(self.posX, self.posY)
        glEnd()

    def process(self, time):
        oldDist = self.dist
        self.dist -= time * ROCK_SPEED
        self.posX  = math.cos(self.angle) * self.dist
        self.posY  = math.sin(self.angle) * self.dist

        for i in range(len(self.game.shells)):
            shell = self.game.shells[i]
            segment = shell.getGuardianSegment(self.angle, self.dist, oldDist)
            if segment is not None:
                if segment.state == segment.STATE_ALIVE:
                    segment.state = segment.STATE_DEAD
                    if i <= self.size:
                        self.state = self.STATE_DEAD
                        return

        if self.dist < 0:
            self.state = self.STATE_DEAD
            self.game.energy = 0
            if self.game.energy < DEAD_ENERGY:
                self.game.gameState = self.game.STATE_GAMEOVER


def normalize(v3):
    len = math.sqrt(v3[0] * v3[0] +
                    v3[1] * v3[1] +
                    v3[2] * v3[2])
    if len == 0: assert(0)
    return (v3[0] / len,
            v3[1] / len,
            v3[2] / len)


def clampCircle(angle):
    pi2 = math.pi * 2.0
    while angle <    0: angle += pi2
    while angle >= pi2: angle -= pi2
    return angle


def clampCircleAround0(angle):
    angle += math.pi
    pi2 = math.pi * 2.0
    while angle <    0: angle += pi2
    while angle >= pi2: angle -= pi2
    return angle - math.pi


def blendVectors4(v1, v2, alpha):
    return (v1[0] + (v2[0] - v1[0]) * alpha,
            v1[1] + (v2[1] - v1[1]) * alpha,
            v1[2] + (v2[2] - v1[2]) * alpha,
            v1[3] + (v2[3] - v1[3]) * alpha)
          
if __name__ == '__main__':
    main()
