import os, sys, time
import pygame
from pygame.locals import *
from Numeric import *

from resources import gamedata


# if you're executing the game from the sources change this to 1 for fullscreen
fullscreen = 0

GRAVITATION = 0.004 # pixels / ms^2

class Animation:
    def __init__(self, frames, speed):
        self.anim_speed = speed
        self.anim_frame = 0.0
        self.anim = frames

    def update(self, ticks):
        new_frame = self.anim_frame + self.anim_speed * ticks
        if int(new_frame) > int(self.anim_frame):
            if int(new_frame) >= len(self.anim):
                new_frame %= len(self.anim)

        self.anim_frame = new_frame
        return self.get_frame()

    def get_frame(self):
        return self.anim[int(self.anim_frame)]
 
class Background:
    def __init__(self, map, tiles, surface):
        self.tiles = tiles
        self.tile_w, self.tile_h = tiles[0].get_size()
        self.map = map
        self.coord = [0.0, 0.0]
        self.scroll_speed = .05
        self.max_x = (len(self.map[1,:]) * self.tile_w) - surface.get_width()

    def get_scroll_coord(self):
        return int(self.coord[0]), int(self.coord[1])

    def to_map_coords(self, coord):
        return (coord[0] // self.tile_w, coord[1] // self.tile_h)

    def to_pixel_coords(self, coord):
        return (coord[0] * self.tile_w, coord[1] * self.tile_h)
        
    def update(self, ticks):
        new_x = self.coord[0] + self.scroll_speed * ticks
        if new_x < self.max_x:
            self.coord[0] = new_x
        else:
            return 1
        return 0
        
    def draw(self, surface):
        coord = int(self.coord[0]), int(self.coord[1])
        left, top = self.to_map_coords(coord)
        dx, dy = self.tile_w, self.tile_h
        x, y = -(coord[0] % dx), -(coord[1] % dy)

        width = (surface.get_width() - x + dx - 1) // dx
        height = (surface.get_height() - y + dy - 1) // dy

        for line in self.map[top:top+height+1, left:left+width+1]:
            x2 = x
            for tile in line:
                surface.blit(self.tiles[tile], (x2, y))
                x2 += dx
            y += dy    

        
class Angel(pygame.sprite.Sprite):
    def __init__(self, clamp_rect):
        pygame.sprite.Sprite.__init__(self)
        self.image = gamedata.angel_right
        self.rect = self.image.get_rect()
        self.clamp = clamp_rect
        self.orientation = [0,0]
        self.speed = 0.3

    def update(self, ticks):
        delta = self.speed*ticks
        self.rect.move_ip(self.orientation[0]*delta, self.orientation[1]*delta)
        self.rect.clamp_ip(self.clamp)

        if self.orientation[0] < 0:
            self.image = gamedata.angel_left
        elif self.orientation[0] > 0:
            self.image = gamedata.angel_right

    def handle_event(self, event):
        if event.type is KEYDOWN:
            if event.key == K_LEFT:
                self.orientation[0] = -1
            elif event.key == K_RIGHT:
                self.orientation[0] = 1
            elif event.key == K_UP:
                self.orientation[1] = -1
            elif event.key == K_DOWN:
                self.orientation[1] = 1            
            
        elif event.type is KEYUP and event.key in (K_LEFT, K_RIGHT, K_UP, K_DOWN):
            if event.key == K_LEFT and self.orientation[0] == -1:
                self.orientation[0] = 0
            elif event.key == K_RIGHT and self.orientation[0] == 1:
                self.orientation[0] = 0
            elif event.key == K_UP and self.orientation[1] == -1:
                self.orientation[1] = 0
            elif event.key == K_DOWN and self.orientation[1] == 1:
                self.orientation[1] = 0

class StupidGuy(pygame.sprite.Sprite):
    def __init__(self):
        pygame.sprite.Sprite.__init__(self)
        self.anim = Animation(gamedata.stupid_guy_anim, .006)
        self.image = self.anim.get_frame()
        self.rect = self.image.get_rect().move(250, 260)


    def they_killed_kenny(self):
        gamedata.snd_dead.play()
        self.image = gamedata.stupid_guy_dead
        self.rect = self.image.get_rect().move(250, 388)
        self.anim = Animation([self.image], 0)

    def update(self, ticks):
        self.image = self.anim.update(ticks)

# proxy for collision tests
class StupidGuyProxy(pygame.sprite.Sprite):
    def __init__(self):
        pygame.sprite.Sprite.__init__(self)
        self.image = gamedata.stupid_guy_a
        self.rect = self.image.get_rect().move(250, 260).inflate(-20,-20)

class Flowers(pygame.sprite.Sprite):
    def __init__(self, background, init_pos):
        pygame.sprite.Sprite.__init__(self)
        self.image = gamedata.flowers
        self.rect = self.image.get_rect()
        self.background = background
        self.map_pos = list(init_pos)
        self.falling = 0
        self.pushed = 0
        self.falling_speed = .0
        self.horizontal_speed = .2
        self.done = 0

    def update(self, ticks):
        if self.falling:
            self.falling_speed += GRAVITATION * ticks
            self.map_pos[1] += self.falling_speed * ticks

        if self.pushed:
            self.map_pos[0] += self.horizontal_speed * ticks

        if self.map_pos[1]>440:
            self.kill()

        self.rect = self.image.get_rect()
        coord = self.background.get_scroll_coord()
        self.rect.move_ip(self.map_pos[0]-coord[0], self.map_pos[1]-coord[1])

        if not self.done and not self.falling and self.rect.left<300:
            self.falling = 1
            gamedata.snd_flowers.play()

    def push(self):
        self.falling = 1
        self.pushed = 1
        gamedata.snd_flowers.play()

    def killing_done(self):
        self.falling = 0
        self.pushed = 0
        self.done = 1
        self.map_pos[1] = 320

class BasicEnemy(pygame.sprite.Sprite):
    def __init__(self):
        pygame.sprite.Sprite.__init__(self)
        self.ass_kicked = 0

    def update(self,ticks):
        if self.ass_kicked:
            self.speed[1] += GRAVITATION * ticks
            self.rect.move_ip(self.speed[0] * ticks, self.speed[1] * ticks)
        else:
            self.rect.move_ip(self.speed * ticks, 0)

    def kick_ass(self):
        self.ass_kicked = 1
        self.speed = [.4, -1]
        self.dead_sound.play()

    def killing_done(self):
        self.speed = 0
        self.ass_kicked = 0 # should already be 0 but just in case
        self.dead_sound.play()

class Dog(BasicEnemy):
    def __init__(self):
        BasicEnemy.__init__(self)
        self.anim = Animation(gamedata.dog_anim, .006)
        self.image = self.anim.get_frame()
        self.rect = self.image.get_rect().move(-200, 300)
        self.speed = 0.15
        gamedata.snd_dog.play()
        self.dead_sound = gamedata.snd_dog

    def update(self, ticks):
        BasicEnemy.update(self, ticks)
        self.image = self.anim.update(ticks)

    def kick_ass(self):
        BasicEnemy.kick_ass(self)
        self.anim = Animation(gamedata.dog_anim, .015)


class Skater(BasicEnemy):
    def __init__(self):
        BasicEnemy.__init__(self)
        self.image = gamedata.skater
        self.rect = self.image.get_rect().move(650, 280)
        self.speed = -0.2
        gamedata.snd_skater.play()
        self.dead_sound = gamedata.snd_skater_dead

class GuardianAngel:

    def __init__(self, fullscreen):
        pygame.mixer.pre_init(44100)
        pygame.init()
        self.fullscreen = fullscreen
        if fullscreen:
            self.buffer = pygame.display.set_mode((640,480), DOUBLEBUF | FULLSCREEN | HWSURFACE, 16)
        else:
            self.buffer = pygame.display.set_mode((640,480), 0)

        pygame.display.set_caption("Guardian Angel")
        pygame.mouse.set_visible(0)
        gamedata.load()

    def display_text(self, message):
        text = gamedata.font.render(message, 1, (0,0,0))
        pos = text.get_rect()
        pos.centerx = self.buffer.get_rect().centerx
        pos.centery = self.buffer.get_rect().centery       
        self.buffer.blit(text, pos)

    def run(self):
        while self.splash():
            for level in gamedata.levels:
                self.play_level(level)
                if not self.win:
                    break
            if not self.no_game_over:
                self.game_over()

    def splash(self):
        self.buffer.blit(gamedata.splash, (0,0))
        pygame.display.flip()
        return self.wait_or_key(1000000)
        
    def game_over(self):
        self.render()
        if self.win:
            self.display_text("Congratulations! You beat the game!")
        else:        
            self.display_text("Game Over")
        pygame.display.flip()
        time.sleep(2)
        self.wait_or_key(6)

    def pre_level(self, level):
        self.background.draw(self.buffer)
        
        self.display_text(level['title'])
        pygame.display.flip()
        time.sleep(.5)
        self.wait_or_key(2)

    def wait_or_key(self, time_to_wait):
        for event in pygame.event.get():
            pass
        time_waited = 0
        while time_waited < time_to_wait:
            for event in pygame.event.get():
                if event.type is QUIT:
                    return 0
                elif event.type is KEYDOWN and event.key==K_ESCAPE:
                    return 0
                elif event.type is KEYDOWN:
                    return 1
            time.sleep(0.1)
            time_waited += 0.1
        return 1


    def play_level(self, level):
        self.init_level(level)
        self.pre_level(level)

        clock = pygame.time.Clock()
        fps = 0
        self.level_over = 0
        self.win = 0
        self.no_game_over = 0
        while not self.level_over:
            ticks = clock.tick(100)
            self.update(ticks)
            if not self.handle_events():
                return 0

            fps = (fps + clock.get_fps()) / 2      
            pygame.display.flip()
        return 1

    def handle_events(self):            
        for event in pygame.event.get():
            if event.type is QUIT:
                self.no_game_over = 1
                return 0
            elif event.type is KEYDOWN and event.key == K_ESCAPE:
                self.no_game_over = 1
                return 0
            elif event.type is KEYDOWN and event.key == K_SPACE:
                self.angelic_intervention()
            else:
                self.angel.handle_event(event)

        return 1

    def update(self, ticks):
        if self.background.update(ticks):
            self.level_over = 1
            self.win = 1
        
        self.sprites.update(ticks)

        killer = pygame.sprite.spritecollideany(self.stupid_guy_proxy, self.danger)
        if killer:
            self.stupid_guy.they_killed_kenny()
            self.background.scroll_speed = 0
            killer.killing_done()
            killer.remove(self.danger)
            self.level_over = 1
            self.sprites.update(ticks) # one last update

        if len(self.dog_schedule) and self.background.get_scroll_coord()[0]>self.dog_schedule[0]:
            self.dog_schedule.pop(0)
            newdog = Dog()
            newdog.add((self.danger, self.bad_guys, self.sprites))

        if len(self.skater_schedule) and self.background.get_scroll_coord()[0]>self.skater_schedule[0]:
            self.skater_schedule.pop(0)
            newskater = Skater()
            newskater.add((self.danger, self.bad_guys, self.sprites))

        self.render()

    def render(self):
        self.background.draw(self.buffer)
        self.bad_guys.draw(self.buffer)
        self.good_guys.draw(self.buffer)

    def init_level(self, level):
        self.background = Background(level['map'], gamedata.tiles, self.buffer)

        self.dog_schedule = level['dogs'][:]
        self.skater_schedule = level['skaters'][:]
        self.angel = Angel(self.buffer.get_rect().inflate(64,64))
        self.stupid_guy = StupidGuy()
        self.stupid_guy_proxy = StupidGuyProxy()
        flowers = [Flowers(self.background, coords) for coords in level['flowers']]

        self.danger = pygame.sprite.Group(flowers)
        self.bad_guys = pygame.sprite.RenderPlain(flowers)
        self.good_guys = pygame.sprite.RenderPlain([self.stupid_guy, self.angel])
        self.sprites = pygame.sprite.Group(flowers+[self.stupid_guy, self.angel])

    def angelic_intervention(self):
        gamedata.snd_angel_action.play()
        collide_list = pygame.sprite.spritecollide(self.angel, self.sprites, 0)
        for sprite in collide_list:
            if isinstance(sprite, StupidGuy):
                gamedata.snd_what.play()
            elif isinstance(sprite, Flowers):
                sprite.push()
            elif isinstance(sprite, BasicEnemy):
                sprite.kick_ass()
                sprite.remove(self.danger)

if __name__=='__main__':
    game = GuardianAngel(fullscreen)
    game.run()
